# Comprehensive Performance Optimization Plan

## Executive Summary

This document outlines a comprehensive performance optimization strategy for the EPOE system, addressing database performance, query optimization, caching, Supabase integration, and scalability concerns.

## 1️⃣ Database Performance Optimization

### Current Issues Identified

1. **N+1 Query Problems:**
   - SubmissionController: Multiple `whereHas()` calls causing subqueries
   - ReportController: Loading full relationships when only summaries needed
   - ClassController: Loading all enrollments and submissions on show page

2. **Missing Indexes:**
   - Composite indexes for common filter combinations
   - Indexes on foreign keys used in WHERE clauses
   - Indexes on status columns for filtering

3. **Inefficient Joins:**
   - Using `whereHas()` instead of direct joins
   - Multiple nested subqueries

### Optimization Strategy

#### A. Database Indexes

**Priority Indexes to Add:**

```sql
-- Composite indexes for common query patterns
CREATE INDEX idx_poe_submissions_status_class ON poe_submissions(status, class_id);
CREATE INDEX idx_poe_submissions_student_status ON poe_submissions(student_id, status);
CREATE INDEX idx_classes_term_department ON classes(term_id, department_id);
CREATE INDEX idx_classes_term_level ON classes(term_id, level_id);
CREATE INDEX idx_enrollments_class_status ON enrollments(class_id, status);
CREATE INDEX idx_poe_evidence_submission ON poe_evidence(poe_submission_id, file_type);

-- Covering indexes for common SELECT patterns
CREATE INDEX idx_submissions_listing ON poe_submissions(class_id, status, submitted_at DESC) 
  INCLUDE (student_id, unit_id, version);

-- Full-text search indexes (PostgreSQL)
CREATE INDEX idx_users_search ON users USING gin(to_tsvector('english', name || ' ' || email));
CREATE INDEX idx_units_search ON units USING gin(to_tsvector('english', name || ' ' || code));
```

#### B. Query Optimization

**Replace `whereHas()` with Joins:**

```php
// BEFORE (Slow - uses subquery)
$query->whereHas('schoolClass', function($q) {
    $q->where('term_id', $termId);
});

// AFTER (Fast - uses JOIN)
$query->join('classes', 'poe_submissions.class_id', '=', 'classes.id')
      ->where('classes.term_id', $termId)
      ->select('poe_submissions.*');
```

**Use Select Specific Columns:**

```php
// BEFORE
$submissions = PoeSubmission::with(['student', 'unit', 'schoolClass'])->get();

// AFTER
$submissions = PoeSubmission::select('id', 'student_id', 'unit_id', 'class_id', 'status', 'submitted_at')
    ->with([
        'student:id,name,email',
        'unit:id,name,code',
        'schoolClass:id,name,term_id'
    ])
    ->get();
```

### C. Term-Based Filtering Optimization

**Global Scope for Active Term:**

```php
// In AppServiceProvider
PoeSubmission::addGlobalScope('activeTerm', function ($query) {
    $activeTermId = ActiveTermService::getActiveTermId();
    if ($activeTermId) {
        $query->join('classes', 'poe_submissions.class_id', '=', 'classes.id')
              ->where('classes.term_id', $activeTermId);
    }
});
```

## 2️⃣ Data Fetching Strategy

### A. Separate List vs Detail Views

**List Views (Index Pages):**
- Load only summary data
- Use `select()` to limit columns
- Lazy load relationships only when needed
- Use `withCount()` instead of loading full relationships

**Detail Views (Show Pages):**
- Load full data only when viewing single record
- Use pagination for related data (evidence, reviews)
- Load evidence files metadata separately

### B. Pagination Strategy

```php
// Use cursor pagination for large datasets
$submissions = PoeSubmission::cursorPaginate(20);

// Or use simple pagination with optimized queries
$submissions = PoeSubmission::select('id', 'student_id', 'unit_id', 'class_id', 'status')
    ->with(['student:id,name', 'unit:id,name'])
    ->simplePaginate(20);
```

### C. Lazy Loading for Evidence

```php
// Load evidence count in list view
$submissions = PoeSubmission::withCount('evidence')->get();

// Load evidence files only in detail view
$submission->load(['evidence' => function($query) {
    $query->select('id', 'poe_submission_id', 'file_name', 'file_type', 'file_size')
          ->orderBy('created_at');
}]);
```

## 3️⃣ Supabase Integration Optimization

### A. Connection Strategy

**Recommended Configuration:**

```env
# Use Session Pooler for better connection management
DB_CONNECTION=pgsql
DB_HOST=aws-0-af-south-1.pooler.supabase.com
DB_PORT=6543
DB_DATABASE=postgres
DB_USERNAME=postgres.xxxxx
DB_PASSWORD=your_password

# Connection Pool Settings
DB_POOL_SIZE=20
DB_MAX_CONNECTIONS=100
```

**Benefits:**
- Better connection reuse
- Reduced connection overhead
- Automatic connection pooling
- Better for concurrent requests

### B. Row Level Security (RLS)

**Recommendation:**
- **Disable RLS for backend/admin access** - Use application-level authorization
- **Enable RLS for direct database access** - If using Supabase client libraries
- **Use service role key** for backend operations

```sql
-- Disable RLS for admin operations (if needed)
ALTER TABLE poe_submissions DISABLE ROW LEVEL SECURITY;
```

### C. Storage Optimization

**Use Supabase Storage URLs:**

```php
// Store only file path/reference, not file content
$evidence->file_path = 'poe-evidence/' . $file->hashName();
$evidence->storage_url = Storage::disk('supabase')->url($evidence->file_path);

// Access files via CDN URLs, not loading into memory
$url = Storage::disk('supabase')->url($evidence->file_path);
```

## 4️⃣ Caching & State Management

### A. Caching Strategy

**Cache Layers:**

1. **Application Cache (Redis/File):**
   - Terms (1 hour)
   - Departments (1 hour)
   - Levels (1 hour)
   - Units (1 hour)
   - Active Term (5 minutes)

2. **Query Result Cache:**
   - Dashboard statistics (5 minutes)
   - User role counts (5 minutes)
   - Submission statistics (5 minutes)

3. **View Cache:**
   - Static dropdown options (1 hour)
   - Filter options (30 minutes)

**Cache Keys:**

```php
'active_term' => 'Active term (5 min TTL)'
'terms.all' => 'All terms (1 hour)'
'departments.all' => 'All departments (1 hour)'
'units.department.{id}' => 'Units by department (1 hour)'
'classes.term.{id}' => 'Classes by term (30 min)'
'stats.dashboard' => 'Dashboard stats (5 min)'
```

### B. Cache Invalidation

**Automatic Invalidation:**

```php
// When term is activated
ActiveTermService::clearCache();
cache()->forget('active_term');
cache()->forget('classes.term.*');

// When class is created/updated
cache()->forget('classes.term.' . $class->term_id);
cache()->forget('stats.dashboard');
```

### C. Term Switching Strategy

**Dynamic Term Switching:**

```php
// Service to handle term switching
class ActiveTermService {
    public static function switchTerm(Term $term) {
        // Close previous active term
        Term::where('status', 'active')
            ->where('id', '!=', $term->id)
            ->update(['status' => 'completed']);
        
        // Activate new term
        $term->update(['status' => 'active']);
        
        // Clear all term-related caches
        Cache::tags(['terms', 'classes', 'submissions'])->flush();
        
        // Warm cache with new term data
        self::warmCache($term);
    }
    
    protected static function warmCache(Term $term) {
        // Pre-load common queries
        cache()->remember('classes.term.' . $term->id, 1800, function() use ($term) {
            return SchoolClass::where('term_id', $term->id)->get();
        });
    }
}
```

## 5️⃣ Scalability & Architecture

### A. Background Jobs

**Queue Heavy Operations:**

```php
// File processing
ProcessEvidenceFile::dispatch($evidence);

// Report generation
GenerateReport::dispatch($reportType, $filters);

// Email notifications
SendSubmissionNotification::dispatch($submission);

// Cache warming
WarmTermCache::dispatch($term);
```

**Queue Configuration:**

```env
QUEUE_CONNECTION=database
# Or use Redis for better performance
QUEUE_CONNECTION=redis
```

### B. Database Partitioning (Future)

For very large datasets:

```sql
-- Partition poe_submissions by term_id
CREATE TABLE poe_submissions_2025_term1 PARTITION OF poe_submissions
    FOR VALUES IN (1);

CREATE TABLE poe_submissions_2025_term2 PARTITION OF poe_submissions
    FOR VALUES IN (2);
```

### C. API Response Shaping

**Use API Resources:**

```php
// SubmissionResource for list view
class SubmissionSummaryResource extends JsonResource {
    public function toArray($request) {
        return [
            'id' => $this->id,
            'student' => $this->student->name,
            'unit' => $this->unit->code,
            'status' => $this->status,
            'evidence_count' => $this->evidence_count,
        ];
    }
}

// Full SubmissionResource for detail view
class SubmissionResource extends JsonResource {
    public function toArray($request) {
        return [
            'id' => $this->id,
            'student' => new UserResource($this->student),
            'unit' => new UnitResource($this->unit),
            'evidence' => EvidenceResource::collection($this->evidence),
            // ... full data
        ];
    }
}
```

## 6️⃣ User Experience Optimization

### A. Loading States

```blade
<!-- Skeleton loading -->
<div class="animate-pulse">
    <div class="h-4 bg-gray-200 rounded w-3/4"></div>
</div>

<!-- Progressive loading -->
<div x-data="{ loading: true }" x-init="fetchData().then(() => loading = false)">
    <div x-show="loading">Loading...</div>
    <div x-show="!loading"><!-- Content --></div>
</div>
```

### B. Lazy Loading Images

```blade
<img src="{{ $evidence->storage_url }}" 
     loading="lazy" 
     alt="{{ $evidence->file_name }}">
```

### C. Infinite Scroll

```javascript
// Load more submissions on scroll
window.addEventListener('scroll', () => {
    if (window.innerHeight + window.scrollY >= document.body.offsetHeight - 1000) {
        loadMoreSubmissions();
    }
});
```

## 7️⃣ Implementation Priority

### Phase 1: Critical (Immediate)
1. ✅ Add composite database indexes
2. ✅ Replace `whereHas()` with joins
3. ✅ Implement selective column loading
4. ✅ Add term-based global scope
5. ✅ Optimize submission queries

### Phase 2: High Priority (Week 1)
1. Implement Redis caching
2. Add query result caching
3. Optimize evidence file loading
4. Implement lazy loading for relationships

### Phase 3: Medium Priority (Week 2)
1. Background job processing
2. API resource shaping
3. Progressive loading UI
4. Cache warming strategies

### Phase 4: Future Enhancements
1. Database partitioning
2. CDN integration
3. Full-text search optimization
4. Advanced analytics caching

## 8️⃣ Expected Performance Improvements

### Before Optimization:
- Dashboard: 3-5 seconds
- Submissions List: 2-4 seconds
- User List: 1-2 seconds
- Database Queries: 50-100+ per page
- Memory Usage: High (loading full objects)

### After Optimization:
- Dashboard: < 500ms (cached) / < 1s (uncached)
- Submissions List: < 800ms
- User List: < 400ms
- Database Queries: 5-15 per page
- Memory Usage: Reduced by 60-70%

## 9️⃣ Monitoring & Metrics

### Key Metrics to Track:
1. Page load time
2. Database query count
3. Cache hit rate
4. Memory usage
5. Response time percentiles (p50, p95, p99)

### Tools:
- Laravel Debugbar (development)
- New Relic / Datadog (production)
- Supabase Analytics
- Application logs

