<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

/**
 * Create the public/storage symlink when PHP symlink() is disabled (e.g. on some cPanel hosts).
 * Run: php artisan storage:link-safe
 */
class StorageLinkFallback extends Command
{
    protected $signature = 'storage:link-safe
                {--force : Recreate the link if it already exists}';

    protected $description = 'Create storage link (fallback when symlink() is disabled on server)';

    public function handle(): int
    {
        $link = public_path('storage');
        $target = storage_path('app/public');

        if (! is_dir($target)) {
            File::ensureDirectoryExists($target);
        }

        if (file_exists($link)) {
            if (! $this->option('force')) {
                $this->error("The [{$link}] path already exists. Use --force to replace.");
                return self::FAILURE;
            }
            if (is_link($link)) {
                @unlink($link);
            } elseif (is_dir($link)) {
                $this->error("The [{$link}] is a directory. Remove it manually and run again, or use a different approach.");
                return self::FAILURE;
            }
        }

        // 1) Try PHP symlink()
        if (function_exists('symlink')) {
            if (@symlink($target, $link)) {
                $this->info("The [{$link}] link has been connected to [{$target}].");
                return self::SUCCESS;
            }
        }

        // 2) Try shell ln -s (works when PHP symlink is disabled but shell allows it)
        if (PHP_OS_FAMILY !== 'Windows' && $this->runShellLink($target, $link)) {
            $this->info("The [{$link}] link has been created via shell.");
            return self::SUCCESS;
        }

        // 3) Manual instructions
        $this->warn('Could not create the storage link automatically (symlink is disabled on this server).');
        $this->line('');
        $this->line('Create it manually:');
        $this->line('  1. SSH:  ln -s ' . $target . ' ' . $link);
        $this->line('  2. Or in cPanel File Manager:');
        $this->line('     - Go to public_html (or your app public folder).');
        $this->line('     - Create a new "Symbolic Link" named "storage" pointing to:');
        $this->line('       ' . $target);
        $this->line('');

        return self::SUCCESS;
    }

    private function runShellLink(string $target, string $link): bool
    {
        $targetReal = realpath($target);
        if ($targetReal === false || ! is_dir($targetReal)) {
            return false;
        }
        $cmd = 'ln -s ' . escapeshellarg($targetReal) . ' ' . escapeshellarg($link);
        @exec($cmd);
        return is_link($link);
    }
}
