<?php

namespace App\Exceptions;

use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Database\QueryException;
use PDOException;
use Throwable;

class Handler extends ExceptionHandler
{
    /**
     * A list of exception types with their corresponding custom log levels.
     *
     * @var array<class-string<\Throwable>, \Psr\Log\LogLevel::*>
     */
    protected $levels = [
        //
    ];

    /**
     * A list of the exception types that are not reported.
     *
     * @var array<int, class-string<\Throwable>>
     */
    protected $dontReport = [
        //
    ];

    /**
     * A list of the inputs that are never flashed to the session on validation exceptions.
     *
     * @var array<int, string>
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     *
     * @return void
     */
    public function register()
    {
        $this->reportable(function (Throwable $e) {
            //
        });
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Throwable  $e
     * @return \Symfony\Component\HttpFoundation\Response
     *
     * @throws \Throwable
     */
    public function render($request, Throwable $e)
    {
        // Handle database connection errors
        if ($this->isDatabaseConnectionError($e)) {
            return $this->renderDatabaseConnectionError($request, $e);
        }

        return parent::render($request, $e);
    }

    /**
     * Check if the exception is a database connection error.
     *
     * @param  \Throwable  $e
     * @return bool
     */
    protected function isDatabaseConnectionError(Throwable $e): bool
    {
        // Check for QueryException with connection errors
        if ($e instanceof QueryException) {
            $message = $e->getMessage();
            $code = $e->getCode();
            
            // Common connection error patterns
            $connectionErrors = [
                'could not translate host name',
                'could not connect to server',
                'connection refused',
                'connection timed out',
                'unknown host',
                'network is unreachable',
                'no route to host',
                'SQLSTATE[08006]',
                'SQLSTATE[08001]',
                'SQLSTATE[HY000]',
            ];

            foreach ($connectionErrors as $error) {
                if (stripos($message, $error) !== false) {
                    return true;
                }
            }
        }

        // Check for PDOException with connection errors
        if ($e instanceof PDOException) {
            $code = $e->getCode();
            // PDO connection error codes
            if (in_array($code, ['08006', '08001', 'HY000', '2002', '2003'])) {
                return true;
            }
        }

        return false;
    }

    /**
     * Render database connection error page.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Throwable  $e
     * @return \Illuminate\Http\Response
     */
    protected function renderDatabaseConnectionError($request, Throwable $e)
    {
        // Log the error for debugging
        \Log::error('Database Connection Error', [
            'message' => $e->getMessage(),
            'code' => $e->getCode(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
        ]);

        // Determine error type
        $errorType = $this->getConnectionErrorType($e);
        $errorMessage = $this->getUserFriendlyMessage($errorType);

        // Return JSON response for API requests
        if ($request->expectsJson() || $request->is('api/*')) {
            return response()->json([
                'error' => 'Database Connection Error',
                'message' => $errorMessage,
                'type' => $errorType,
            ], 503);
        }

        // Return HTML error page
        return response()->view('errors.database-connection', [
            'errorType' => $errorType,
            'errorMessage' => $errorMessage,
            'originalError' => config('app.debug') ? $e->getMessage() : null,
        ], 503);
    }

    /**
     * Get the type of connection error.
     *
     * @param  \Throwable  $e
     * @return string
     */
    protected function getConnectionErrorType(Throwable $e): string
    {
        $message = strtolower($e->getMessage());

        if (stripos($message, 'could not translate host name') !== false || 
            stripos($message, 'unknown host') !== false) {
            return 'dns_resolution';
        }

        if (stripos($message, 'connection refused') !== false) {
            return 'connection_refused';
        }

        if (stripos($message, 'connection timed out') !== false || 
            stripos($message, 'timeout') !== false) {
            return 'timeout';
        }

        if (stripos($message, 'network is unreachable') !== false || 
            stripos($message, 'no route to host') !== false) {
            return 'network_unreachable';
        }

        return 'general';
    }

    /**
     * Get user-friendly error message.
     *
     * @param  string  $errorType
     * @return string
     */
    protected function getUserFriendlyMessage(string $errorType): string
    {
        $messages = [
            'dns_resolution' => 'Unable to connect to the database server. Please check your internet connection and try again.',
            'connection_refused' => 'The database server is not accepting connections. Please check your database configuration.',
            'timeout' => 'The connection to the database server timed out. Please check your internet connection and try again.',
            'network_unreachable' => 'Unable to reach the database server. Please check your internet connection.',
            'general' => 'Unable to connect to the database. Please check your internet connection and database configuration.',
        ];

        return $messages[$errorType] ?? $messages['general'];
    }

    /**
     * Determine if the exception should be reported.
     *
     * @param  \Throwable  $e
     * @return bool
     */
    public function shouldReport(Throwable $e)
    {
        // Don't report database connection errors in production (they're expected during outages)
        if ($this->isDatabaseConnectionError($e) && !config('app.debug')) {
            return false;
        }

        return parent::shouldReport($e);
    }
}
