<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Term;
use App\Models\SchoolClass;
use App\Models\PoeSubmission;
use App\Models\Department;
use App\Models\Unit;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        // Cache statistics for 5 minutes to improve performance
        $stats = cache()->remember('admin.dashboard.stats', 300, function () {
            return [
                'total_users' => User::count(),
                'active_users' => User::where('status', 'active')->count(),
                'total_terms' => Term::count(),
                'active_terms' => Term::where('status', 'active')->count(),
                'total_classes' => SchoolClass::count(),
                'total_departments' => Department::count(),
                'total_units' => Unit::count(),
                'total_submissions' => PoeSubmission::count(),
                'pending_submissions' => PoeSubmission::whereIn('status', ['submitted', 'under_review'])->count(),
                'approved_submissions' => PoeSubmission::where('status', 'validator_approved')->count(),
                'rejected_submissions' => PoeSubmission::where('status', 'rejected')->count(),
            ];
        });

        // Users by role - single query
        $usersByRole = cache()->remember('admin.dashboard.users_by_role', 300, function () {
            return User::selectRaw('role, count(*) as count')
                ->groupBy('role')
                ->get()
                ->pluck('count', 'role');
        });

        // Recent submissions - optimized with term filtering (using whereHas to avoid join conflicts)
        $activeTermId = ActiveTermService::getActiveTermId();
        $recentSubmissions = cache()->remember('admin.dashboard.recent_submissions', 300, function() use ($activeTermId) {
            $query = PoeSubmission::select([
                'poe_submissions.id',
                'poe_submissions.student_id',
                'poe_submissions.unit_id',
                'poe_submissions.class_id',
                'poe_submissions.status',
                'poe_submissions.submitted_at',
                'poe_submissions.created_at',
            ])
            ->with([
                'student:id,name,email',
                'unit:id,name,code',
                'schoolClass:id,name'
            ]);
            
            // Use whereHas instead of join to avoid conflict with eager loading
            if ($activeTermId) {
                $query->whereHas('schoolClass', function($q) use ($activeTermId) {
                    $q->where('term_id', $activeTermId);
                });
            }
            
            return $query->latest('poe_submissions.created_at')->limit(10)->get();
        });

        // Recent activity
        $recentTerms = Term::select('id', 'name', 'start_date', 'end_date', 'status', 'created_at')
            ->latest()
            ->limit(5)
            ->get();
            
        $recentClasses = SchoolClass::with(['term:id,name', 'department:id,name', 'level:id,name'])
            ->select('id', 'term_id', 'department_id', 'level_id', 'name', 'created_at')
            ->latest()
            ->limit(5)
            ->get();

        return view('admin.dashboard', compact(
            'stats',
            'usersByRole',
            'recentSubmissions',
            'recentTerms',
            'recentClasses'
        ));
    }
}
