<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Term;
use App\Models\SchoolClass;
use App\Models\PoeSubmission;
use App\Models\PracticalEvidence;
use App\Models\User;
use App\Models\Unit;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class HistoryController extends Controller
{
    /**
     * Display history index - list of all past terms
     */
    public function index()
    {
        $terms = Term::whereIn('status', ['completed', 'archived'])
            ->orderBy('start_date', 'desc')
            ->get();

        return view('admin.history.index', compact('terms'));
    }

    /**
     * Display details of a specific historical term
     */
    public function show(Term $term)
    {
        // Only allow viewing completed or archived terms
        if ($term->isActive() || $term->isDraft()) {
            return redirect()->route('admin.history.index')
                ->with('error', 'You can only view completed or archived terms in history.');
        }

        $term->load([
            'classes.department',
            'classes.level',
            'termDepartmentUnits.unit',
            'termDepartmentUnits.department'
        ]);

        // Get all class IDs for this term
        $classIds = $term->classes()->pluck('id')->toArray();
        
        // Get all unit IDs for this term
        $unitIds = DB::table('term_department_units')
            ->where('term_id', $term->id)
            ->distinct()
            ->pluck('unit_id')
            ->toArray();

        // Calculate comprehensive statistics for this term
        $stats = [
            'total_classes' => count($classIds),
            'total_units' => count($unitIds),
            'total_students' => (int) DB::table('enrollments')
                ->whereIn('class_id', $classIds)
                ->where('status', 'active')
                ->selectRaw('COUNT(DISTINCT student_id) as count')
                ->value('count') ?? 0,
            'total_trainers' => (int) DB::table('trainer_unit_class')
                ->whereIn('class_id', $classIds)
                ->whereIn('unit_id', $unitIds)
                ->selectRaw('COUNT(DISTINCT trainer_id) as count')
                ->value('count') ?? 0,
            'total_submissions' => DB::table('poe_submissions')
                ->whereIn('class_id', $classIds)
                ->count(),
            'total_practical_evidence' => DB::table('practical_evidences')
                ->whereIn('class_id', $classIds)
                ->count(),
            'total_portfolio_evidence' => DB::table('poe_submissions')
                ->whereIn('class_id', $classIds)
                ->count() + DB::table('practical_evidences')
                ->whereIn('class_id', $classIds)
                ->count(),
        ];

        return view('admin.history.show', compact('term', 'stats'));
    }

    /**
     * Display classes for a historical term
     */
    public function classes(Term $term)
    {
        if ($term->isActive() || $term->isDraft()) {
            return redirect()->route('admin.history.index')
                ->with('error', 'You can only view completed or archived terms in history.');
        }

        $classes = $term->classes()
            ->with(['department', 'level'])
            ->orderBy('name')
            ->get();

        return view('admin.history.classes', compact('term', 'classes'));
    }

    /**
     * Display units for a historical term
     */
    public function units(Term $term)
    {
        if ($term->isActive() || $term->isDraft()) {
            return redirect()->route('admin.history.index')
                ->with('error', 'You can only view completed or archived terms in history.');
        }

        $unitIds = DB::table('term_department_units')
            ->where('term_id', $term->id)
            ->distinct()
            ->pluck('unit_id');

        $units = Unit::whereIn('id', $unitIds)
            ->with('department')
            ->orderBy('name')
            ->get();

        return view('admin.history.units', compact('term', 'units'));
    }

    /**
     * Display submissions for a historical term
     */
    public function submissions(Term $term, Request $request)
    {
        if ($term->isActive() || $term->isDraft()) {
            return redirect()->route('admin.history.index')
                ->with('error', 'You can only view completed or archived terms in history.');
        }

        $classIds = $term->classes()->pluck('id');

        $query = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('class_id', $classIds)
            ->with(['student', 'unit', 'schoolClass']);

        // Apply filters
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('unit') && $request->unit) {
            $query->where('unit_id', $request->unit);
        }

        if ($request->has('class') && $request->class) {
            $query->where('class_id', $request->class);
        }

        $submissions = $query->latest('created_at')->paginate(20);

        $units = Unit::whereIn('id', function($q) use ($term) {
            $q->select('unit_id')
                ->from('term_department_units')
                ->where('term_id', $term->id);
        })->orderBy('name')->get();

        $classes = SchoolClass::whereIn('id', $classIds)->orderBy('name')->get();

        return view('admin.history.submissions', compact('term', 'submissions', 'units', 'classes'));
    }

    /**
     * Display trainers for a historical term
     */
    public function trainers(Term $term)
    {
        if ($term->isActive() || $term->isDraft()) {
            return redirect()->route('admin.history.index')
                ->with('error', 'You can only view completed or archived terms in history.');
        }

        $classIds = $term->classes()->pluck('id')->toArray();

        $unitIds = DB::table('term_department_units')
            ->where('term_id', $term->id)
            ->distinct()
            ->pluck('unit_id')
            ->toArray();

        $trainerIds = DB::table('trainer_unit_class')
            ->whereIn('class_id', $classIds)
            ->whereIn('unit_id', $unitIds)
            ->distinct()
            ->pluck('trainer_id')
            ->toArray();

        $trainers = User::whereIn('id', $trainerIds)
            ->where('role', 'trainer')
            ->with('department')
            ->orderBy('name')
            ->get()
            ->map(function($trainer) use ($classIds, $unitIds) {
                $trainer->units_count = DB::table('trainer_unit_class')
                    ->where('trainer_id', $trainer->id)
                    ->whereIn('class_id', $classIds)
                    ->whereIn('unit_id', $unitIds)
                    ->distinct('unit_id')
                    ->count('unit_id');
                return $trainer;
            });

        return view('admin.history.trainers', compact('term', 'trainers', 'classIds', 'unitIds'));
    }

    /**
     * Display practical evidence for a historical term
     */
    public function practicalEvidence(Term $term, Request $request)
    {
        if ($term->isActive() || $term->isDraft()) {
            return redirect()->route('admin.history.index')
                ->with('error', 'You can only view completed or archived terms in history.');
        }

        $classIds = $term->classes()->pluck('id')->toArray();

        $query = PracticalEvidence::whereIn('class_id', $classIds)
            ->with(['student', 'unit', 'schoolClass']);

        // Apply filters
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('unit') && $request->unit) {
            $query->where('unit_id', $request->unit);
        }

        if ($request->has('class') && $request->class) {
            $query->where('class_id', $request->class);
        }

        $evidences = $query->latest('uploaded_at')->paginate(20);

        $units = Unit::whereIn('id', function($q) use ($term) {
            $q->select('unit_id')
                ->from('term_department_units')
                ->where('term_id', $term->id);
        })->orderBy('name')->get();

        $classes = SchoolClass::whereIn('id', $classIds)->orderBy('name')->get();

        return view('admin.history.practical-evidence', compact('term', 'evidences', 'units', 'classes'));
    }

    /**
     * Print unit marks for a historical term
     */
    public function printUnitMarks(Term $term, $unitId)
    {
        if ($term->isActive() || $term->isDraft()) {
            return redirect()->route('admin.history.index')
                ->with('error', 'You can only view completed or archived terms in history.');
        }

        $unit = Unit::findOrFail($unitId);
        $classIds = $term->classes()->pluck('id')->toArray();

        // Get all assignments for this unit
        $allAssignments = Assignment::where('unit_id', $unitId)
            ->with('trainer:id,name')
            ->orderBy('type')
            ->orderBy('created_at')
            ->get();

        // Get assessment structure
        $assessmentStructure = $unit->getAssessmentStructure();

        // Group assignments by type
        $groupedAssignments = [
            'theory' => [],
            'practical' => [],
            'oral' => [],
            'project' => [],
        ];

        foreach ($allAssignments as $assignment) {
            if (isset($groupedAssignments[$assignment->type])) {
                $groupedAssignments[$assignment->type][] = $assignment;
            }
        }

        // Get class IDs with this unit
        $unitClassIds = DB::table('class_unit')
            ->where('unit_id', $unitId)
            ->whereIn('class_id', $classIds)
            ->pluck('class_id')
            ->toArray();

        // Get all students
        $students = User::where('role', 'student')
            ->whereHas('enrollments', function($q) use ($unitClassIds) {
                $q->whereIn('class_id', $unitClassIds)
                  ->where('status', 'active');
            })
            ->orderBy('name')
            ->get();

        // Get marks for all students
        $marks = [];
        foreach ($students as $student) {
            $studentMarks = [];
            foreach ($allAssignments as $assignment) {
                $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->where('student_id', $student->id)
                    ->first();

                $mark = 0;
                if ($submission && $submission->marks) {
                    $mark = $submission->marks;
                } elseif ($submission && $submission->assessorToolMarks()->count() > 0) {
                    $mark = $submission->assessorToolMarks()->sum('marks_obtained');
                }

                $studentMarks[$assignment->id] = $mark;
            }
            $marks[$student->id] = $studentMarks;
        }

        // Calculate category averages
        $categoryAverages = [];
        foreach ($students as $student) {
            $averages = [];
            foreach (['theory', 'practical', 'oral', 'project'] as $type) {
                $typeAssignments = $groupedAssignments[$type] ?? [];
                if (count($typeAssignments) > 0) {
                    $total = 0;
                    $count = 0;
                    foreach ($typeAssignments as $assignment) {
                        $mark = $marks[$student->id][$assignment->id] ?? 0;
                        $total += $mark;
                        $count++;
                    }
                    $averages[$type] = $count > 0 ? round($total / $count, 2) : 0;
                } else {
                    $averages[$type] = 0;
                }
            }
            $categoryAverages[$student->id] = $averages;
        }

        // Get institution settings
        $institutionName = Setting::get('institution_name', '');
        $institutionAddress = Setting::get('institution_address', '');
        $institutionPhone = Setting::get('institution_phone', '');
        $institutionEmail = Setting::get('institution_email', '');

        return view('admin.history.print-unit-marks', compact(
            'term', 'unit', 'students', 'allAssignments', 'groupedAssignments',
            'assessmentStructure', 'marks', 'categoryAverages',
            'institutionName', 'institutionAddress', 'institutionPhone', 'institutionEmail'
        ));
    }
}

