<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\SchoolClass;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PracticalMarksController extends Controller
{
    /**
     * Display compiled practical marks for an assignment
     */
    public function index(Request $request, Assignment $assignment)
    {
        if ($assignment->type !== 'practical') {
            abort(404, 'This is not a practical assignment.');
        }

        // Get all students enrolled in classes that have this unit assigned
        $unit = $assignment->unit;
        $classIds = DB::table('class_unit')
            ->where('unit_id', $unit->id)
            ->pluck('class_id')
            ->toArray();

        // Get students enrolled in these classes
        $students = DB::table('enrollments')
            ->whereIn('class_id', $classIds)
            ->where('status', 'active')
            ->join('users', 'enrollments.student_id', '=', 'users.id')
            ->select('users.id', 'users.name', 'users.email', 'users.admission_number', 'enrollments.class_id')
            ->orderBy('users.name')
            ->get();

        // Get submissions and marks for each student
        $compiledMarks = [];
        foreach ($students as $student) {
            $submission = \App\Models\AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();

            $marks = 0;
            $markingStatus = 'not_marked';
            
            if ($submission) {
                $markingStatus = $submission->marking_status ?? 'not_marked';
                if ($submission->marks) {
                    $marks = $submission->marks;
                } elseif ($submission->assessorToolMarks()->count() > 0) {
                    $marks = $submission->assessorToolMarks()->sum('marks_obtained');
                }
            }

            $compiledMarks[] = [
                'student' => $student,
                'submission' => $submission,
                'marks' => $marks,
                'marking_status' => $markingStatus,
                'class' => SchoolClass::find($student->class_id),
            ];
        }

        // Filter by status
        $statusFilter = $request->get('status', 'all');
        if ($statusFilter !== 'all') {
            $compiledMarks = array_filter($compiledMarks, function($item) use ($statusFilter) {
                return $item['marking_status'] === $statusFilter;
            });
        }

        // Sort
        $sortBy = $request->get('sort', 'name');
        if ($sortBy === 'marks') {
            usort($compiledMarks, function($a, $b) {
                return $b['marks'] <=> $a['marks'];
            });
        } else {
            usort($compiledMarks, function($a, $b) {
                return strcmp($a['student']->name, $b['student']->name);
            });
        }

        // Statistics
        $stats = [
            'total_students' => count($compiledMarks),
            'marked' => count(array_filter($compiledMarks, fn($item) => $item['marking_status'] === 'marked')),
            'not_marked' => count(array_filter($compiledMarks, fn($item) => $item['marking_status'] === 'not_marked')),
            'absent' => count(array_filter($compiledMarks, fn($item) => $item['marking_status'] === 'absent')),
        ];

        return view('admin.practical-marks.index', compact(
            'assignment',
            'unit',
            'compiledMarks',
            'stats',
            'statusFilter',
            'sortBy'
        ));
    }

    /**
     * Lock results
     */
    public function lockResults(Assignment $assignment)
    {
        $assignment->update([
            'results_locked' => true,
        ]);

        return redirect()->back()->with('success', 'Results have been locked.');
    }

    /**
     * Unlock results
     */
    public function unlockResults(Assignment $assignment)
    {
        $assignment->update([
            'results_locked' => false,
        ]);

        return redirect()->back()->with('success', 'Results have been unlocked.');
    }

    /**
     * Release results to students
     */
    public function releaseResults(Assignment $assignment)
    {
        $assignment->update([
            'results_released' => true,
            'results_released_at' => now(),
        ]);

        return redirect()->back()->with('success', 'Results have been released to students.');
    }

    /**
     * Export practical marks
     */
    public function export(Assignment $assignment, $format)
    {
        if ($assignment->type !== 'practical') {
            abort(404);
        }

        // Same export logic as trainer
        $unit = $assignment->unit;
        $classIds = DB::table('class_unit')
            ->where('unit_id', $unit->id)
            ->pluck('class_id')
            ->toArray();

        $students = DB::table('enrollments')
            ->whereIn('class_id', $classIds)
            ->where('status', 'active')
            ->join('users', 'enrollments.student_id', '=', 'users.id')
            ->select('users.id', 'users.name', 'users.email', 'users.admission_number', 'enrollments.class_id')
            ->orderBy('users.name')
            ->get();

        $compiledMarks = [];
        foreach ($students as $student) {
            $submission = \App\Models\AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();

            $marks = 0;
            $markingStatus = 'not_marked';
            
            if ($submission) {
                $markingStatus = $submission->marking_status ?? 'not_marked';
                if ($submission->marks) {
                    $marks = $submission->marks;
                } elseif ($submission->assessorToolMarks()->count() > 0) {
                    $marks = $submission->assessorToolMarks()->sum('marks_obtained');
                }
            }

            $compiledMarks[] = [
                'student' => $student,
                'marks' => $marks,
                'marking_status' => $markingStatus,
                'class' => SchoolClass::find($student->class_id),
            ];
        }

        usort($compiledMarks, function($a, $b) {
            return strcmp($a['student']->name, $b['student']->name);
        });

        if ($format === 'pdf') {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('admin.practical-marks.export-pdf', [
                'assignment' => $assignment,
                'unit' => $unit,
                'compiledMarks' => $compiledMarks,
            ]);
            
            $filename = 'Practical_Marks_' . $assignment->title . '_' . now()->format('Y-m-d') . '.pdf';
            return $pdf->download($filename);
        }

        abort(404, 'Invalid export format');
    }
}
