<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Models\User;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class UnitAllocationController extends Controller
{
    /**
     * Display a listing of unit allocations.
     */
    public function index(Request $request)
    {
        $activeTermId = ActiveTermService::getActiveTermId();

        $query = DB::table('trainer_unit_class')
            ->join('users', 'trainer_unit_class.trainer_id', '=', 'users.id')
            ->join('units', 'trainer_unit_class.unit_id', '=', 'units.id')
            ->join('classes', 'trainer_unit_class.class_id', '=', 'classes.id')
            ->join('departments', 'units.department_id', '=', 'departments.id')
            ->select(
                'trainer_unit_class.id',
                'trainer_unit_class.trainer_id',
                'trainer_unit_class.unit_id',
                'trainer_unit_class.class_id',
                'users.name as trainer_name',
                'users.email as trainer_email',
                'units.name as unit_name',
                'units.code as unit_code',
                'classes.name as class_name',
                'classes.code as class_code',
                'departments.name as department_name',
                'trainer_unit_class.created_at'
            );

        // By default, show allocations for the active term only (to match HOD view)
        if ($activeTermId) {
            $query->where('classes.term_id', $activeTermId);
        }

        // Filter by department
        if ($request->has('department') && $request->department) {
            $query->where('departments.id', $request->department);
        }

        // Filter by class
        if ($request->has('class') && $request->class) {
            $query->where('classes.id', $request->class);
        }

        // Filter by trainer
        if ($request->has('trainer') && $request->trainer) {
            $query->where('users.id', $request->trainer);
        }

        // Filter by unit
        if ($request->has('unit') && $request->unit) {
            $query->where('units.id', $request->unit);
        }

        // Filter by active term
        $activeTermId = ActiveTermService::getActiveTermId();
        if ($activeTermId) {
            $query->where('classes.term_id', $activeTermId);
        }

        $allocations = $query->orderBy('trainer_unit_class.created_at', 'desc')->paginate(20);

        // Get filter options
        $departments = \App\Models\Department::orderBy('name')->get();
        $activeTerm = ActiveTermService::getActiveTerm();
        $classes = $activeTermId 
            ? SchoolClass::where('term_id', $activeTermId)->orderBy('name')->get()
            : collect();
        $trainers = User::where('role', 'trainer')->where('status', 'active')->orderBy('name')->get();
        $units = Unit::orderBy('name')->get();

        return view('admin.unit-allocations.index', compact(
            'allocations',
            'departments',
            'classes',
            'trainers',
            'units',
            'activeTerm'
        ));
    }

    /**
     * Show the form for creating a new allocation.
     */
    public function create(Request $request)
    {
        $activeTermId = ActiveTermService::getActiveTermId();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        $departments = \App\Models\Department::orderBy('name')->get();
        $classes = $activeTermId 
            ? SchoolClass::where('term_id', $activeTermId)->with('department')->orderBy('name')->get()
            : collect();
        $trainers = User::where('role', 'trainer')
            ->where('status', 'active')
            ->with('department')
            ->orderBy('name')
            ->get();
        // Get all units with their classes relationship - ensure newly added units are included
        // Pass as collection, let the view handle the conversion
        $units = Unit::with(['department', 'classes'])
            ->orderBy('name')
            ->get();

        // Pre-select if provided
        $selectedDepartment = $request->get('department');
        $selectedClass = $request->get('class');
        $selectedTrainer = $request->get('trainer');
        $selectedUnit = $request->get('unit');

        return view('admin.unit-allocations.create', compact(
            'departments',
            'classes',
            'trainers',
            'units',
            'activeTerm',
            'selectedDepartment',
            'selectedClass',
            'selectedTrainer',
            'selectedUnit'
        ));
    }

    /**
     * Store a newly created allocation.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'trainer_id' => 'required|exists:users,id',
            'unit_ids' => 'required|array|min:1',
            'unit_ids.*' => 'required|exists:units,id',
            'class_id' => 'required|exists:classes,id',
        ], [
            'trainer_id.required' => 'Please select a trainer.',
            'trainer_id.exists' => 'The selected trainer does not exist.',
            'unit_ids.required' => 'Please select at least one unit.',
            'unit_ids.array' => 'Units must be provided as an array.',
            'unit_ids.min' => 'Please select at least one unit.',
            'unit_ids.*.required' => 'Invalid unit selected.',
            'unit_ids.*.exists' => 'One or more selected units do not exist.',
            'class_id.required' => 'Please select a class.',
            'class_id.exists' => 'The selected class does not exist.',
        ]);

        // Custom validation: Check if trainer is actually a trainer
        $trainer = User::find($request->trainer_id);
        if (!$trainer || $trainer->role !== 'trainer') {
            $validator->errors()->add('trainer_id', 'The selected user is not a trainer.');
        }

        // Get class
        $class = SchoolClass::find($request->class_id);
        if (!$class) {
            $validator->errors()->add('class_id', 'Invalid class selected.');
        }
        
        // Get term_id from the class (required for trainer_unit_class)
        $termId = $class ? $class->term_id : null;
        if (!$termId) {
            $validator->errors()->add('class_id', 'The selected class does not have a term assigned.');
        }

        // Get units and validate each one
        $unitIds = $request->unit_ids ?? [];
        $units = Unit::whereIn('id', $unitIds)->get();
        
        $errors = [];
        $validUnits = [];
        
        foreach ($units as $unit) {
            // Check if unit is assigned to class
            if (!$class->units()->where('units.id', $unit->id)->exists()) {
                $errors[] = "Unit '{$unit->name}' is not assigned to the selected class. Please assign it first.";
                continue;
            }
            
            // Check if trainer and unit are in same department
            if ($trainer && $trainer->department_id !== $unit->department_id) {
                $errors[] = "Unit '{$unit->name}' must be in the same department as the trainer.";
                continue;
            }
            
            // Check for duplicate assignment (same trainer, same unit, same class)
            $duplicateExists = DB::table('trainer_unit_class')
                ->where('trainer_id', $request->trainer_id)
                ->where('unit_id', $unit->id)
                ->where('class_id', $request->class_id)
                ->exists();
            
            if ($duplicateExists) {
                $errors[] = "Trainer is already assigned to unit '{$unit->name}' in this class.";
                continue;
            }
            
            // Check if another trainer is already assigned to this unit in this class
            $otherTrainerExists = DB::table('trainer_unit_class')
                ->where('unit_id', $unit->id)
                ->where('class_id', $request->class_id)
                ->where('trainer_id', '!=', $request->trainer_id)
                ->exists();
            
            if ($otherTrainerExists) {
                $existingTrainer = DB::table('trainer_unit_class')
                    ->join('users', 'trainer_unit_class.trainer_id', '=', 'users.id')
                    ->where('trainer_unit_class.unit_id', $unit->id)
                    ->where('trainer_unit_class.class_id', $request->class_id)
                    ->where('trainer_unit_class.trainer_id', '!=', $request->trainer_id)
                    ->value('users.name');
                
                $errors[] = "Another trainer ({$existingTrainer}) is already assigned to unit '{$unit->name}' in this class.";
                continue;
            }
            
            $validUnits[] = $unit;
        }
        
        // Add custom errors to validator
        foreach ($errors as $error) {
            $validator->errors()->add('unit_ids', $error);
        }

        if ($validator->fails() || !empty($errors)) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        if (empty($validUnits)) {
            return redirect()->back()
                ->withErrors(['unit_ids' => 'No valid units to allocate. Please check the errors above.'])
                ->withInput();
        }

        try {
            $allocations = [];
            $now = now();
            
            foreach ($validUnits as $unit) {
                $allocations[] = [
                    'trainer_id' => $request->trainer_id,
                    'unit_id' => $unit->id,
                    'class_id' => $request->class_id,
                    'term_id' => $termId,
                    'created_at' => $now,
                    'updated_at' => $now,
                ];
            }
            
            // Insert all allocations at once
            DB::table('trainer_unit_class')->insert($allocations);
            
            // Log audit for each allocation
            try {
                $unitNames = collect($validUnits)->pluck('name')->join(', ');
                \App\Models\AuditLog::create([
                    'user_id' => auth()->id(),
                    'action' => 'created',
                    'model_type' => 'trainer_unit_class',
                    'model_id' => null,
                    'description' => "Assigned trainer {$trainer->name} to " . count($validUnits) . " unit(s) ({$unitNames}) in class {$class->name}",
                    'created_at' => now(),
                ]);
            } catch (\Exception $auditException) {
                // Log audit error but don't fail the allocation
                \Log::warning('Failed to create audit log for unit allocation', [
                    'error' => $auditException->getMessage(),
                ]);
            }

            $count = count($validUnits);
            $message = $count === 1 
                ? 'Trainer successfully assigned to unit in class.'
                : "Trainer successfully assigned to {$count} units in class.";

            return redirect()->route('admin.unit-allocations.index')
                ->with('success', $message);
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle database-specific errors
            $errorMessage = 'Failed to assign trainer. ';
            
            // Check for unique constraint violation
            if (str_contains($e->getMessage(), 'trainer_unit_class_unique') || 
                str_contains($e->getMessage(), 'duplicate key value violates unique constraint')) {
                $errorMessage = 'One or more allocations already exist. Please check the existing allocations.';
            } elseif (str_contains($e->getMessage(), 'foreign key')) {
                $errorMessage = 'Invalid trainer, unit, or class selected.';
            } else {
                $errorMessage .= 'Please check that the units are assigned to the class and try again.';
            }
            
            \Log::error('Unit allocation creation failed', [
                'error' => $e->getMessage(),
                'request' => $request->all(),
            ]);
            
            return redirect()->back()
                ->withErrors(['unit_ids' => $errorMessage])
                ->with('error', $errorMessage)
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Unit allocation creation failed', [
                'error' => $e->getMessage(),
                'request' => $request->all(),
            ]);
            
            return redirect()->back()
                ->with('error', 'Failed to assign trainer: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove an allocation.
     */
    public function destroy($id)
    {
        $allocation = DB::table('trainer_unit_class')
            ->join('users', 'trainer_unit_class.trainer_id', '=', 'users.id')
            ->join('units', 'trainer_unit_class.unit_id', '=', 'units.id')
            ->join('classes', 'trainer_unit_class.class_id', '=', 'classes.id')
            ->where('trainer_unit_class.id', $id)
            ->select(
                'trainer_unit_class.*',
                'users.name as trainer_name',
                'units.name as unit_name',
                'classes.name as class_name'
            )
            ->first();

        if (!$allocation) {
            return redirect()->route('admin.unit-allocations.index')
                ->with('error', 'Allocation not found.');
        }

        try {
            DB::table('trainer_unit_class')->where('id', $id)->delete();

            // Log audit
            \App\Models\AuditLog::create([
                'user_id' => auth()->id(),
                'action' => 'deleted',
                'model_type' => 'trainer_unit_class',
                'model_id' => $id,
                'description' => "Removed trainer {$allocation->trainer_name} from unit {$allocation->unit_name} in class {$allocation->class_name}",
            ]);

            return redirect()->route('admin.unit-allocations.index')
                ->with('success', 'Allocation removed successfully.');
        } catch (\Exception $e) {
            return redirect()->route('admin.unit-allocations.index')
                ->with('error', 'Failed to remove allocation. Please try again.');
        }
    }
}
