<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\AssessorToolStructure;
use App\Models\AssessorToolMark;
use App\Services\AssessorToolParserService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class AssessorToolController extends Controller
{
    protected $parserService;

    public function __construct(AssessorToolParserService $parserService)
    {
        $this->parserService = $parserService;
    }

    private function authorizeAssignment(Assignment $assignment): void
    {
        $hod = Auth::user();
        if ($assignment->trainer_id !== $hod->id) {
            abort(403, 'You do not have permission to manage this assignment.');
        }
        $assignment->load('unit:id,department_id');
        if ($assignment->unit->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to manage this assignment.');
        }
    }

    public function createStructure(Assignment $assignment)
    {
        $this->authorizeAssignment($assignment);

        if ($assignment->type !== 'practical') {
            return redirect()->route('hod.assignments.show', $assignment)
                ->with('error', 'Assessor tool structure is only available for practical assignments.');
        }

        $structure = AssessorToolStructure::where('assignment_id', $assignment->id)->first();
        $assessorToolFiles = $assignment->assessor_tool ?? [];

        return view('hod.assessor-tool.create-structure', compact('assignment', 'structure', 'assessorToolFiles'));
    }

    public function storeStructure(Request $request, Assignment $assignment)
    {
        $this->authorizeAssignment($assignment);

        if ($assignment->type !== 'practical') {
            return redirect()->back()->with('error', 'Invalid assignment type.');
        }

        $request->validate([
            'parse_method' => ['required', 'in:manual,auto'],
            'structure' => ['required_if:parse_method,manual', 'array'],
            'structure.*.section' => ['nullable', 'string', 'max:255'],
            'structure.*.item_description' => ['required', 'string'],
            'structure.*.marks_available' => ['required', 'numeric', 'min:0'],
        ]);

        DB::beginTransaction();
        try {
            if ($request->parse_method === 'auto') {
                if (!$assignment->assessor_tool || !is_array($assignment->assessor_tool) || empty($assignment->assessor_tool)) {
                    throw new \Exception('No assessor tool found in this assignment. Please ensure the assessor tool was uploaded during practical creation.');
                }

                $assessorToolPath = $assignment->assessor_tool[0] ?? null;
                if (!$assessorToolPath || !Storage::disk('public')->exists($assessorToolPath)) {
                    throw new \Exception('Assessor tool file not found. Please ensure the assessor tool file exists in storage.');
                }

                $extension = pathinfo($assessorToolPath, PATHINFO_EXTENSION);
                $mimeType = match(strtolower($extension)) {
                    'pdf' => 'application/pdf',
                    'doc' => 'application/msword',
                    'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                    default => 'application/pdf',
                };

                $parsedData = $this->parserService->parseDocument($assessorToolPath, $mimeType);

                $structure = AssessorToolStructure::updateOrCreate(
                    ['assignment_id' => $assignment->id],
                    [
                        'structure' => $parsedData,
                        'original_file_path' => $assessorToolPath,
                        'total_marks' => $parsedData['total_marks'],
                        'header_data' => $parsedData['header_data'] ?? [],
                    ]
                );
            } else {
                $structureData = [];
                $totalMarks = 0;

                foreach ($request->structure as $index => $item) {
                    $rowIdentifier = $item['section']
                        ? strtolower(str_replace([' ', ':'], ['_', ''], $item['section'])) . '_item_' . ($index + 1)
                        : 'item_' . ($index + 1);

                    $structureData[] = [
                        'section' => $item['section'] ?? null,
                        'row_identifier' => $rowIdentifier,
                        'item_description' => $item['item_description'],
                        'marks_available' => (float) $item['marks_available'],
                        'display_order' => $index + 1,
                        'is_subtotal' => false,
                        'is_total' => false,
                    ];

                    $totalMarks += (float) $item['marks_available'];
                }

                $structure = AssessorToolStructure::updateOrCreate(
                    ['assignment_id' => $assignment->id],
                    [
                        'structure' => [
                            'header_data' => $request->header_data ?? [],
                            'sections' => $structureData,
                            'total_marks' => $totalMarks,
                        ],
                        'total_marks' => $totalMarks,
                        'header_data' => $request->header_data ?? [],
                    ]
                );
            }

            DB::commit();

            return redirect()->route('hod.assessor-tool.marking', $assignment)
                ->with('success', 'Assessor tool structure created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to create structure: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function marking(Request $request, Assignment $assignment, $submissionId = null)
    {
        $this->authorizeAssignment($assignment);

        if ($assignment->type !== 'practical') {
            abort(404);
        }

        $structure = AssessorToolStructure::where('assignment_id', $assignment->id)->first();

        if (!$structure) {
            return redirect()->route('hod.assessor-tool.create-structure', $assignment)
                ->with('warning', 'Please create the assessor tool structure first.');
        }

        if (!isset($structure->structure) || !is_array($structure->structure)) {
            $structure->structure = ['sections' => [], 'header_data' => [], 'total_marks' => 0];
            $structure->save();
        }

        $assignment->load('unit');
        $unit = $assignment->unit;
        $classId = $assignment->class_id;

        if (!$unit || !$classId) {
            $submissions = collect();
        } else {
            $allStudents = DB::table('enrollments')
                ->where('enrollments.class_id', $classId)
                ->where('enrollments.status', 'active')
                ->join('users', 'enrollments.student_id', '=', 'users.id')
                ->where('users.role', 'student')
                ->select('users.id', 'users.name', 'users.email', 'users.admission_number')
                ->orderBy('users.name')
                ->get();

            $existingSubmissions = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->with(['student:id,name,email,admission_number', 'assessorToolMarks'])
                ->get()
                ->keyBy('student_id');

            $submissions = collect();
            foreach ($allStudents as $student) {
                $sub = $existingSubmissions->get($student->id);
                if ($sub) {
                    $submissions->push($sub);
                } else {
                    $submissions->push((object)[
                        'id' => null,
                        'student_id' => $student->id,
                        'student' => (object)[
                            'id' => $student->id,
                            'name' => $student->name,
                            'email' => $student->email,
                            'admission_number' => $student->admission_number,
                        ],
                        'marking_status' => 'not_marked',
                        'marks' => null,
                    ]);
                }
            }
        }

        $submission = null;
        $marks = collect();
        $studentIdParam = $request->query('student_id');

        $studentId = null;
        if ($studentIdParam) {
            $studentIdParam = trim($studentIdParam);
            if ($studentIdParam !== 'null' && $studentIdParam !== '' && is_numeric($studentIdParam)) {
                $studentId = (int)$studentIdParam;
                if ($studentId <= 0) {
                    $studentId = null;
                }
            }
        }

        $selectedStudentId = $studentId;

        if ($studentId && $studentId > 0) {
            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $studentId)
                ->with('student:id,name,email,admission_number')
                ->first();

            if (!$submission) {
                $submission = AssignmentSubmission::create([
                    'assignment_id' => $assignment->id,
                    'student_id' => $studentId,
                    'status' => 'submitted',
                    'marking_status' => 'not_marked',
                ]);
                $submission->load('student:id,name,email,admission_number');
            }

            $marks = AssessorToolMark::where('assignment_submission_id', $submission->id)
                ->orderBy('display_order')
                ->get()
                ->keyBy('row_identifier');
        } elseif ($submissionId && $submissionId !== 'new' && is_numeric($submissionId)) {
            $submission = AssignmentSubmission::where('id', $submissionId)
                ->where('assignment_id', $assignment->id)
                ->with('student:id,name,email,admission_number')
                ->firstOrFail();

            $marks = AssessorToolMark::where('assignment_submission_id', $submission->id)
                ->orderBy('display_order')
                ->get()
                ->keyBy('row_identifier');
        }

        $marked = $submissions->filter(function($sub) {
            return ($sub->marking_status ?? 'not_marked') === 'marked';
        })->count();
        $absent = $submissions->filter(function($sub) {
            return ($sub->marking_status ?? 'not_marked') === 'absent';
        })->count();
        $notMarked = $submissions->filter(function($sub) {
            return ($sub->marking_status ?? 'not_marked') === 'not_marked';
        })->count();
        $assessed = $marked + $absent;

        $stats = [
            'total' => $submissions->count(),
            'marked' => $marked,
            'not_marked' => $notMarked,
            'absent' => $absent,
            'assessed' => $assessed,
        ];

        return view('hod.assessor-tool.marking', compact(
            'assignment',
            'structure',
            'submissions',
            'submission',
            'marks',
            'selectedStudentId',
            'stats'
        ));
    }

    public function saveMarks(Request $request, Assignment $assignment, AssignmentSubmission $submission = null)
    {
        $this->authorizeAssignment($assignment);

        if (!$submission) {
            $request->validate([
                'student_id' => 'required|exists:users,id',
            ]);

            $submission = AssignmentSubmission::firstOrCreate([
                'assignment_id' => $assignment->id,
                'student_id' => $request->student_id,
            ], [
                'status' => 'submitted',
                'marking_status' => 'not_marked',
            ]);
        } else {
            if ($submission->assignment_id !== $assignment->id) {
                abort(403);
            }
        }

        $structure = AssessorToolStructure::where('assignment_id', $assignment->id)->firstOrFail();
        $structureData = $structure->structure;

        $request->validate([
            'marks' => ['required', 'array'],
            'marks.*.marks_obtained' => ['required', 'numeric', 'min:0'],
            'marks.*.comments' => ['nullable', 'string', 'max:1000'],
        ]);

        DB::beginTransaction();
        try {
            $totalMarksObtained = 0;

            foreach ($request->marks as $rowIdentifier => $markData) {
                $rowData = collect($structureData['sections'] ?? [])
                    ->firstWhere('row_identifier', $rowIdentifier);

                if (!$rowData) {
                    continue;
                }

                $marksAvailable = $rowData['marks_available'] ?? 0;
                $marksObtained = min((float) $markData['marks_obtained'], $marksAvailable);

                AssessorToolMark::updateOrCreate(
                    [
                        'assignment_submission_id' => $submission->id,
                        'row_identifier' => $rowIdentifier,
                    ],
                    [
                        'assignment_id' => $assignment->id,
                        'structure_id' => $structure->id,
                        'section' => $rowData['section'] ?? null,
                        'item_description' => $rowData['item_description'],
                        'marks_available' => $marksAvailable,
                        'marks_obtained' => $marksObtained,
                        'comments' => $markData['comments'] ?? null,
                        'display_order' => $rowData['display_order'] ?? 0,
                    ]
                );

                $totalMarksObtained += $marksObtained;
            }

            $wasAlreadyMarked = $submission->marking_status === 'marked';

            // Invalidate any existing generated document so the next "Generate Document" uses current marks.
            $submission->update([
                'marks' => $totalMarksObtained,
                'status' => 'graded',
                'marking_status' => 'marked',
                'graded_at' => now(),
                'completed_assessor_tool_path' => null,
            ]);
            $documentPath = null;

            if ($documentPath && !$wasAlreadyMarked) {
                try {
                    $student = $submission->student;
                    $student->notify(new \App\Notifications\PracticalMarksGraded(
                        $assignment,
                        $submission,
                        $totalMarksObtained,
                        $documentPath
                    ));
                } catch (\Exception $e) {
                    \Log::error('Failed to send notification to student: ' . $e->getMessage());
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Edits have been noted and updated in the assessor tool with marks.',
                'total_marks' => $totalMarksObtained,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to save marks: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function generateDocument(Assignment $assignment, AssignmentSubmission $submission)
    {
        $this->authorizeAssignment($assignment);

        if ($submission->assignment_id !== $assignment->id) {
            abort(403, 'Submission does not belong to this assignment.');
        }

        // If the completed document already exists, reuse it.
        $documentPath = $submission->completed_assessor_tool_path;
        if ($documentPath && \Illuminate\Support\Facades\Storage::disk('public')->exists($documentPath)) {
            return \Illuminate\Support\Facades\Storage::disk('public')->download($documentPath);
        }

        // Otherwise, generate and store it once, then download.
        $structure = AssessorToolStructure::where('assignment_id', $assignment->id)->firstOrFail();
        $marks = AssessorToolMark::where('assignment_submission_id', $submission->id)
            ->orderBy('display_order')
            ->get();

        $generator = new \App\Services\AssessorToolGeneratorService();
        $documentPath = $generator->generateAndStore($structure, $marks, $submission);

        if (!$documentPath || !\Illuminate\Support\Facades\Storage::disk('public')->exists($documentPath)) {
            abort(500, 'Failed to generate assessor tool document.');
        }

        $submission->update([
            'completed_assessor_tool_path' => $documentPath,
        ]);

        return \Illuminate\Support\Facades\Storage::disk('public')->download($documentPath);
    }
}
