<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\Message;
use Illuminate\Support\Facades\Auth;
use Illuminate\Notifications\DatabaseNotification;

class MessageController extends Controller
{
    /**
     * Display a listing of the student's notifications and messages.
     */
    public function index()
    {
        $user = Auth::user();

        // Get Laravel notifications (unread first, then by date)
        $notifications = $user->notifications()
            ->orderBy('read_at', 'asc')
            ->orderBy('created_at', 'desc')
            ->paginate(15, ['*'], 'notifications_page');
        
        // Load sender information for notifications that have sender_id
        $senderIds = $notifications->getCollection()
            ->pluck('data.sender_id')
            ->filter()
            ->unique();
        
        $senders = \App\Models\User::whereIn('id', $senderIds)
            ->select('id', 'name', 'role', 'email')
            ->get()
            ->keyBy('id');
        
        // Create a map of notification IDs to senders for the view
        $notificationSenders = [];
        $notifications->getCollection()->each(function($notification) use ($senders, &$notificationSenders) {
            $senderId = $notification->data['sender_id'] ?? null;
            if ($senderId && $senders->has($senderId)) {
                $notificationSenders[$notification->id] = $senders->get($senderId);
            }
        });

        // Get direct messages (unread first, then by date)
        $messages = Message::where('receiver_id', $user->id)
            ->with(['sender:id,name,email,role'])
            ->orderBy('is_read', 'asc')
            ->orderBy('created_at', 'desc')
            ->paginate(15, ['*'], 'messages_page');

        // Count unread items
        $unreadNotifications = $user->unreadNotifications()->count();
        $unreadMessages = Message::where('receiver_id', $user->id)
            ->where('is_read', false)
            ->count();

        return view('student.messages.index', compact(
            'notifications',
            'messages',
            'unreadNotifications',
            'unreadMessages',
            'notificationSenders'
        ));
    }

    /**
     * Mark a notification as read.
     */
    public function markNotificationAsRead($notificationId)
    {
        $user = Auth::user();
        $notification = $user->notifications()->find($notificationId);

        if ($notification && !$notification->read_at) {
            $notification->markAsRead();
        }

        return redirect()->back()->with('success', 'Notification marked as read.');
    }

    /**
     * Mark a message as read.
     */
    public function markMessageAsRead($messageId)
    {
        $user = Auth::user();
        $message = Message::where('receiver_id', $user->id)->findOrFail($messageId);

        if (!$message->is_read) {
            $message->markAsRead();
        }

        return redirect()->back()->with('success', 'Message marked as read.');
    }

    /**
     * Mark all notifications as read.
     */
    public function markAllNotificationsAsRead()
    {
        $user = Auth::user();
        $user->unreadNotifications->markAsRead();

        return redirect()->back()->with('success', 'All notifications marked as read.');
    }

    /**
     * Mark all messages as read.
     */
    public function markAllMessagesAsRead()
    {
        $user = Auth::user();
        Message::where('receiver_id', $user->id)
            ->where('is_read', false)
            ->update([
                'is_read' => true,
                'read_at' => now(),
            ]);

        return redirect()->back()->with('success', 'All messages marked as read.');
    }

    /**
     * Show a specific notification.
     */
    public function showNotification($notificationId)
    {
        $user = Auth::user();
        $notification = $user->notifications()->findOrFail($notificationId);

        // Load sender information if available
        $senderId = $notification->data['sender_id'] ?? null;
        $sender = null;
        if ($senderId) {
            $sender = \App\Models\User::select('id', 'name', 'role', 'email', 'department_id')
                ->with('department:id,name')
                ->find($senderId);
        }
        
        // Mark as read when viewing
        if (!$notification->read_at) {
            $notification->markAsRead();
        }
        
        // Note: sender is already passed separately to the view
        // We don't need to store it in notification data since it's passed as a separate variable

        return view('student.messages.show-notification', compact('notification', 'sender'));
    }

    /**
     * Show a specific message.
     */
    public function showMessage($messageId)
    {
        $user = Auth::user();
        $message = Message::where('receiver_id', $user->id)
            ->with(['sender:id,name,email,role,department_id' => ['department:id,name']])
            ->findOrFail($messageId);

        // Mark as read when viewing
        if (!$message->is_read) {
            $message->markAsRead();
        }

        return view('student.messages.show-message', compact('message'));
    }
}
