<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\PoeEvidence;
use App\Models\PracticalEvidence;
use App\Models\PracticalEvidenceTask;
use App\Models\Unit;
use App\Models\SchoolClass;
use App\Models\Enrollment;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SubmissionController extends Controller
{
    /**
     * Display a listing of the student's practical evidence.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        // Get student's practical evidence for the active term
        $query = PracticalEvidence::where('student_id', $user->id)
            ->with(['unit:id,name,code', 'schoolClass:id,name,code']);

        // Filter by status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter by unit
        if ($request->unit) {
            $query->where('unit_id', $request->unit);
        }

        // Filter by class
        if ($request->class) {
            $query->where('class_id', $request->class);
        }

        // Search
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->whereHas('unit', function($subQ) use ($request) {
                    $subQ->where('name', 'like', '%' . $request->search . '%')
                          ->orWhere('code', 'like', '%' . $request->search . '%');
                })
                ->orWhere('task_name', 'like', '%' . $request->search . '%');
            });
        }

        // Scope to active term
        if ($activeTerm) {
            $query->where('term_id', $activeTerm->id);
        }

        $allEvidences = $query->latest('uploaded_at')->get();

        // Group evidence by student_id, unit_id, class_id, and task_name
        $groupedEvidences = $allEvidences->groupBy(function($evidence) {
            return $evidence->student_id . '_' . $evidence->unit_id . '_' . $evidence->class_id . '_' . $evidence->task_name;
        });

        // Get student's enrolled classes for the active term
        $enrolledClasses = $user->enrollments()
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                if ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                }
            })
            ->with(['schoolClass:id,name,code'])
            ->get()
            ->pluck('schoolClass');

        // Get units assigned to student's classes
        $assignedUnits = Unit::whereHas('classes', function($q) use ($user, $activeTerm) {
            $classIds = $user->enrollments()
                ->whereHas('schoolClass', function($query) use ($activeTerm) {
                    if ($activeTerm) {
                        $query->where('term_id', $activeTerm->id);
                    }
                })
                ->pluck('class_id');
            $q->whereIn('classes.id', $classIds);
        })
        ->with('department:id,name')
        ->orderBy('name')
        ->get();

        // Statistics
        $baseQuery = PracticalEvidence::where('student_id', $user->id);
        if ($activeTerm) {
            $baseQuery->where('term_id', $activeTerm->id);
        }

        $stats = [
            'total' => (clone $baseQuery)->count(),
            'submitted' => (clone $baseQuery)->where('status', 'submitted')->count(),
            'reviewed' => (clone $baseQuery)->where('status', 'reviewed')->count(),
            'accepted' => (clone $baseQuery)->where('status', 'accepted')->count(),
            'rejected' => (clone $baseQuery)->where('status', 'rejected')->count(),
        ];

        return view('student.submissions.index', compact(
            'groupedEvidences',
            'activeTerm',
            'stats',
            'enrolledClasses',
            'assignedUnits'
        ));
    }

    /**
     * Show the form for creating a new submission.
     */
    public function create(Request $request)
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return redirect()->route('student.submissions.index')
                ->with('error', 'No active term found. Please contact your administrator.');
        }

        // Get student's enrolled classes for the active term
        $enrolledClasses = $user->enrollments()
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['schoolClass:id,name,code,department_id,level_id', 'schoolClass.department:id,name', 'schoolClass.level:id,name'])
            ->get()
            ->pluck('schoolClass');

        // Get units assigned to student's classes
        $classIds = $enrolledClasses->pluck('id');
        $assignedUnits = Unit::whereHas('classes', function($q) use ($classIds) {
            $q->whereIn('classes.id', $classIds);
        })
        ->with('department:id,name')
        ->orderBy('name')
        ->get();

        // Pre-select unit if provided
        $selectedUnit = $request->unit ? Unit::find($request->unit) : null;
        $selectedClass = $request->class ? SchoolClass::find($request->class) : null;

        return view('student.submissions.create', compact(
            'enrolledClasses',
            'assignedUnits',
            'activeTerm',
            'selectedUnit',
            'selectedClass'
        ));
    }

    /**
     * Store a newly created submission.
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return redirect()->back()
                ->with('error', 'No active term found. Please contact your administrator.')
                ->withInput();
        }

        $request->validate([
            'unit_id' => ['required', 'exists:units,id'],
            'class_id' => ['required', 'exists:classes,id'],
            'task_name' => ['required', 'string', 'max:255'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'files' => ['required', 'array', 'min:1'],
            'files.*' => ['file', 'max:10240', 'mimes:pdf,doc,docx,jpg,jpeg,png,mp4,mp3,wav'],
        ], [
            'files.required' => 'Please select at least one file to upload.',
            'files.*.max' => 'One or more files exceed the 10MB limit per file. Please choose smaller files.',
            'files.*.mimes' => 'One or more files have an invalid type. Allowed: PDF, DOC, DOCX, JPG, JPEG, PNG, MP4, MP3, WAV.',
        ]);

        // Verify student is enrolled in the class
        $enrollment = $user->enrollments()
            ->where('class_id', $request->class_id)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->first();

        if (!$enrollment) {
            return redirect()->back()
                ->with('error', 'You are not enrolled in the selected class.')
                ->withInput();
        }

        // Verify unit is assigned to the class
        $unit = Unit::findOrFail($request->unit_id);
        if (!$unit->classes()->where('classes.id', $request->class_id)->exists()) {
            return redirect()->back()
                ->with('error', 'The selected unit is not assigned to the selected class.')
                ->withInput();
        }

        DB::beginTransaction();
        try {
            // Create practical evidence records for each file
            $uploadedEvidences = [];
            
            foreach ($request->file('files') as $file) {
                // Determine file type
                $extension = strtolower($file->getClientOriginalExtension());
                $evidenceType = $this->getEvidenceType($extension);
                
                // Create storage path: practical-evidence/term_{id}/unit_{id}/student_{id}/
                $storagePath = sprintf(
                    'practical-evidence/term_%s/unit_%s/student_%s',
                    $activeTerm->id,
                    $request->unit_id,
                    $user->id
                );
                
                // Store file
                $path = $file->store($storagePath, 'public');
                
                // Create practical evidence record
                $evidence = PracticalEvidence::create([
                    'student_id' => $user->id,
                    'unit_id' => $request->unit_id,
                    'class_id' => $request->class_id,
                    'term_id' => $activeTerm->id,
                    'evidence_type' => $evidenceType,
                    'file_path' => $path,
                    'file_name' => basename($path),
                    'original_name' => $file->getClientOriginalName(),
                    'file_size' => $file->getSize(),
                    'mime_type' => $file->getMimeType(),
                    'task_name' => $request->task_name,
                    'notes' => $request->notes,
                    'status' => 'submitted',
                    'uploaded_at' => now(),
                ]);
                
                $uploadedEvidences[] = $evidence;
            }

            // Create or update task record
            $task = PracticalEvidenceTask::firstOrCreate(
                [
                    'student_id' => $user->id,
                    'unit_id' => $request->unit_id,
                    'class_id' => $request->class_id,
                    'term_id' => $activeTerm->id,
                    'task_name' => $request->task_name,
                ],
                [
                    'status' => 'submitted',
                    'submitted_at' => now(),
                ]
            );

            // If task already existed, update submitted_at if this is the first file
            if ($task->wasRecentlyCreated === false && !$task->submitted_at) {
                $task->update(['submitted_at' => now()]);
            }

            DB::commit();

            return redirect()->route('student.submissions.index')
                ->with('success', 'Practical evidence uploaded successfully. Your trainer will review it.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'An error occurred while uploading the evidence: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified practical evidence.
     */
    public function show($id)
    {
        $user = Auth::user();

        // Try to find as PracticalEvidence first
        $evidence = PracticalEvidence::find($id);
        
        if ($evidence) {
            // Ensure student can only view their own evidence
            if ($evidence->student_id !== $user->id) {
                abort(403, 'Unauthorized access.');
            }

            // Get all evidence for the same task (same student, unit, class, task_name)
            $allTaskEvidence = PracticalEvidence::where('student_id', $evidence->student_id)
                ->where('unit_id', $evidence->unit_id)
                ->where('class_id', $evidence->class_id)
                ->where('task_name', $evidence->task_name)
                ->with(['comments.user:id,name,email'])
                ->orderBy('uploaded_at')
                ->get();

            $evidence->load([
                'unit:id,name,code,description,department_id' => ['department:id,name'],
                'schoolClass:id,name,code,term_id,department_id,level_id' => [
                    'term:id,name,status',
                    'department:id,name',
                    'level:id,name'
                ],
                'comments.user:id,name,email',
            ]);

            return view('student.submissions.show', compact('evidence', 'allTaskEvidence'));
        }

        // Fallback to old POE submission for backward compatibility
        $submission = PoeSubmission::findOrFail($id);
        
        if ($submission->student_id !== $user->id) {
            abort(403, 'Unauthorized access.');
        }

        $submission->load([
            'unit:id,name,code,description,department_id' => ['department:id,name'],
            'schoolClass:id,name,code,term_id,department_id,level_id' => [
                'term:id,name,status',
                'department:id,name',
                'level:id,name'
            ],
            'evidence',
            'reviews:id,poe_submission_id,reviewer_id,marks,feedback,status,reviewed_at' => [
                'reviewer:id,name,email'
            ],
            'validation:id,poe_submission_id,validator_id,status,comments,validated_at' => [
                'validator:id,name,email'
            ]
        ]);

        return view('student.submissions.show', compact('submission'));
    }

    /**
     * Show the form for editing the specified submission.
     */
    public function edit(PoeSubmission $submission)
    {
        $user = Auth::user();

        // Ensure student can only edit their own draft submissions
        if ($submission->student_id !== $user->id) {
            abort(403, 'Unauthorized access.');
        }

        if ($submission->status !== 'draft') {
            return redirect()->route('student.submissions.show', $submission)
                ->with('error', 'Only draft submissions can be edited.');
        }

        $submission->load(['unit', 'schoolClass', 'evidence']);

        return view('student.submissions.edit', compact('submission'));
    }

    /**
     * Update the specified submission.
     */
    public function update(Request $request, PoeSubmission $submission)
    {
        $user = Auth::user();

        // Ensure student can only update their own draft submissions
        if ($submission->student_id !== $user->id) {
            abort(403, 'Unauthorized access.');
        }

        if ($submission->status !== 'draft') {
            return redirect()->back()
                ->with('error', 'Only draft submissions can be updated.');
        }

        $request->validate([
            'notes' => ['nullable', 'string', 'max:1000'],
            'files' => ['nullable', 'array'],
            'files.*' => ['file', 'max:10240', 'mimes:pdf,doc,docx,jpg,jpeg,png,mp4,mp3,wav'],
            'remove_evidence' => ['nullable', 'array'],
            'remove_evidence.*' => ['exists:poe_evidence,id'],
        ]);

        DB::beginTransaction();
        try {
            // Update notes
            $submission->update([
                'notes' => $request->notes,
            ]);

            // Remove evidence if requested
            if ($request->remove_evidence) {
                foreach ($request->remove_evidence as $evidenceId) {
                    $evidence = PoeEvidence::find($evidenceId);
                    if ($evidence && $evidence->poe_submission_id === $submission->id) {
                        if (Storage::exists($evidence->file_path)) {
                            Storage::delete($evidence->file_path);
                        }
                        $evidence->delete();
                    }
                }
            }

            // Add new evidence files
            if ($request->hasFile('files')) {
                foreach ($request->file('files') as $file) {
                    $path = $file->store('poe-evidence', 'public');
                    $extension = $file->getClientOriginalExtension();
                    $fileType = $this->getFileType($extension);

                    PoeEvidence::create([
                        'poe_submission_id' => $submission->id,
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $path,
                        'file_type' => $fileType,
                        'file_size' => $file->getSize(),
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('student.submissions.show', $submission)
                ->with('success', 'Submission updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'An error occurred while updating the submission: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Submit the submission for review.
     */
    public function submit(PoeSubmission $submission)
    {
        $user = Auth::user();

        // Ensure student can only submit their own draft submissions
        if ($submission->student_id !== $user->id) {
            abort(403, 'Unauthorized access.');
        }

        if ($submission->status !== 'draft') {
            return redirect()->back()
                ->with('error', 'Only draft submissions can be submitted for review.');
        }

        // Ensure submission has at least one evidence file
        if ($submission->evidence()->count() === 0) {
            return redirect()->back()
                ->with('error', 'Please add at least one evidence file before submitting.');
        }

        $submission->update([
            'status' => 'submitted',
            'submitted_at' => now(),
        ]);

        return redirect()->route('student.submissions.show', $submission)
            ->with('success', 'Submission submitted for review successfully.');
    }

    /**
     * Delete the specified submission.
     */
    public function destroy(PoeSubmission $submission)
    {
        $user = Auth::user();

        // Ensure student can only delete their own draft submissions
        if ($submission->student_id !== $user->id) {
            abort(403, 'Unauthorized access.');
        }

        if ($submission->status !== 'draft') {
            return redirect()->back()
                ->with('error', 'Only draft submissions can be deleted.');
        }

        // Delete associated evidence files
        foreach ($submission->evidence as $evidence) {
            if (Storage::exists($evidence->file_path)) {
                Storage::delete($evidence->file_path);
            }
            $evidence->delete();
        }

        $submission->delete();

        return redirect()->route('student.submissions.index')
            ->with('success', 'Submission deleted successfully.');
    }

    /**
     * Download evidence file.
     */
    public function downloadEvidence(PoeEvidence $evidence)
    {
        $user = Auth::user();

        // Ensure student can only download evidence from their own submissions
        if ($evidence->poeSubmission->student_id !== $user->id) {
            abort(403, 'Unauthorized access.');
        }

        if (!Storage::exists($evidence->file_path)) {
            return redirect()->back()
                ->with('error', 'File not found.');
        }

        return Storage::download($evidence->file_path, $evidence->file_name);
    }

    /**
     * Get file type from extension.
     */
    private function getFileType($extension)
    {
        $imageTypes = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg'];
        $videoTypes = ['mp4', 'avi', 'mov', 'wmv', 'flv'];
        $audioTypes = ['mp3', 'wav', 'ogg', 'm4a'];
        $documentTypes = ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt'];

        $ext = strtolower($extension);
        
        if (in_array($ext, $imageTypes)) {
            return 'image';
        } elseif (in_array($ext, $videoTypes)) {
            return 'video';
        } elseif (in_array($ext, $audioTypes)) {
            return 'audio';
        } elseif (in_array($ext, $documentTypes)) {
            return 'document';
        }
        
        return 'other';
    }

    /**
     * Get evidence type from extension for practical evidence.
     */
    private function getEvidenceType($extension)
    {
        $imageTypes = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg'];
        $videoTypes = ['mp4', 'avi', 'mov', 'wmv', 'flv'];
        $audioTypes = ['mp3', 'wav', 'ogg', 'm4a'];
        $documentTypes = ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt'];

        $ext = strtolower($extension);
        
        if (in_array($ext, $imageTypes)) {
            return 'image';
        } elseif (in_array($ext, $videoTypes)) {
            return 'video';
        } elseif (in_array($ext, $audioTypes)) {
            return 'audio';
        } elseif (in_array($ext, $documentTypes)) {
            return 'document';
        }
        
        return 'document'; // Default to document
    }
}
