<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\SchoolClass;
use App\Models\PoeSubmission;
use App\Models\User;
use App\Services\ActiveTermService;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index()
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return view('trainer.dashboard', [
                'activeTerm' => null,
                'assignedClasses' => collect(),
                'assignedUnits' => collect(),
                'stats' => [
                    'total_submissions' => 0,
                    'pending_submissions' => 0,
                ],
                'pendingSubmissions' => collect(),
                'pendingReviews' => collect(),
                'recentSubmissions' => collect(),
                'studentPerformance' => [],
                'totalStudents' => 0,
                'statusDistribution' => [],
                'overdueSubmissionsCount' => 0,
                'unitIds' => [],
                'classIds' => [],
            ])->with('warning', 'No active term found. Please contact your administrator.');
        }

        // First, get active term classes for this trainer
        $activeTermClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->pluck('id')
            ->toArray();

        // Get assigned units from trainer_unit_class - ONLY for active term classes
        $unitIds = \Illuminate\Support\Facades\DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->whereIn('class_id', $activeTermClassIds)
            ->distinct()
            ->pluck('unit_id')
            ->toArray();

        // If no units assigned in active term, return empty
        if (empty($unitIds)) {
            return view('trainer.dashboard', [
                'activeTerm' => $activeTerm,
                'assignedUnits' => collect(),
                'stats' => [
                    'total_submissions' => 0,
                    'pending_submissions' => 0,
                ],
                'pendingSubmissions' => collect(),
                'pendingReviews' => collect(),
                'recentSubmissions' => collect(),
                'studentPerformance' => [],
                'totalStudents' => 0,
                'statusDistribution' => [],
                'overdueSubmissionsCount' => 0,
                'unitIds' => [],
                'classIds' => [],
            ])->with('info', 'You don\'t have any units assigned for the current term.');
        }

        // Batch load all unit-class mappings - ONLY for active term classes
        $unitClassMap = \Illuminate\Support\Facades\DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $activeTermClassIds)
            ->get()
            ->groupBy('unit_id')
            ->map(function($items) {
                return $items->pluck('class_id')->toArray();
            });
        
        $allClassIds = $unitClassMap->flatten()->unique()->toArray();
        
        // Batch load classes - already filtered by active term
        $classes = SchoolClass::whereIn('id', $allClassIds)
            ->where('term_id', $activeTerm->id)
            ->select('id', 'name', 'code')
            ->get()
            ->keyBy('id');
        
        // Batch load students count per unit
        $studentsCounts = \Illuminate\Support\Facades\DB::table('enrollments')
            ->join('users', 'enrollments.student_id', '=', 'users.id')
            ->where('users.role', 'student')
            ->whereIn('enrollments.class_id', $allClassIds)
            ->selectRaw('enrollments.class_id, count(distinct enrollments.student_id) as count')
            ->groupBy('enrollments.class_id')
            ->pluck('count', 'class_id');
        
        // Calculate students per unit
        $unitStudentsCount = [];
        foreach ($unitClassMap as $unitId => $classIds) {
            $unitStudentsCount[$unitId] = collect($classIds)->sum(function($classId) use ($studentsCounts) {
                return $studentsCounts[$classId] ?? 0;
            });
        }
        
        // Batch load assignments counts - filtered by active term
        // Only count assignments created during the active term period
        $assignmentsCounts = \App\Models\Assignment::whereIn('unit_id', $unitIds)
            ->where('trainer_id', $trainer->id)
            ->where('created_at', '>=', $activeTerm->start_date)
            ->where('created_at', '<=', $activeTerm->end_date ?? now())
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');
        
        // Batch load submissions counts
        $submissionsCounts = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $allClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');
        
        // Batch load pending submissions counts (replaced reviews)
        $pendingSubmissionsCounts = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $allClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->whereIn('status', ['submitted', 'under_review'])
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');
        
        // Get units with their details
        $assignedUnits = \App\Models\Unit::whereIn('id', $unitIds)
            ->with(['department:id,name,code'])
            ->get()
            ->map(function($unit) use ($unitClassMap, $classes, $unitStudentsCount, $assignmentsCounts, $submissionsCounts, $pendingSubmissionsCounts) {
                $classIds = $unitClassMap[$unit->id] ?? [];
                // Ensure we always return a collection, even if $classIds is empty
                if (empty($classIds)) {
                    $unit->assigned_classes = collect();
                } else {
                    // Use filter since $classes is keyed by ID - filter by checking if class ID is in $classIds
                    $unit->assigned_classes = $classes->filter(function($class) use ($classIds) {
                        return in_array($class->id, $classIds);
                    })->values();
                }
                $unit->students_count = $unitStudentsCount[$unit->id] ?? 0;
                $unit->assignments_count = $assignmentsCounts[$unit->id] ?? 0;
                $unit->submissions_count = $submissionsCounts[$unit->id] ?? 0;
                $unit->pending_submissions_count = $pendingSubmissionsCounts[$unit->id] ?? 0;
                return $unit;
            })
            ->sortBy('name')
            ->values();

        // Get class IDs and unit IDs for filtering - already filtered by active term
        $unitIds = $assignedUnits->pluck('id')->toArray();
        $classIds = $allClassIds; // Already filtered to active term classes

        // Handle empty arrays
        if (empty($classIds) || empty($unitIds)) {
            return view('trainer.dashboard', [
                'activeTerm' => $activeTerm,
                'assignedUnits' => $assignedUnits,
                'stats' => [
                    'total_submissions' => 0,
                    'pending_submissions' => 0,
                ],
                'pendingSubmissions' => collect(),
                'pendingReviews' => collect(),
                'recentSubmissions' => collect(),
                'studentPerformance' => [],
                'totalStudents' => 0,
                'statusDistribution' => [],
                'overdueSubmissionsCount' => 0,
                'unitIds' => [],
                'classIds' => [],
            ])->with('info', 'You don\'t have any units assigned yet.');
        }

        // Get total students in assigned classes
        $totalStudents = User::where('role', 'student')
            ->whereHas('enrollments', function($query) use ($classIds) {
                $query->whereIn('class_id', $classIds);
            })
            ->count();

        // Get submissions from assigned classes and units
        $submissionsQuery = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('class_id', $classIds)
            ->whereIn('unit_id', $unitIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['student:id,name,email', 'unit:id,name,code', 'schoolClass:id,name']);

        // Pending submissions with days calculation - optimized
        $pendingSubmissions = (clone $submissionsQuery)
            ->select('poe_submissions.id', 'poe_submissions.student_id', 'poe_submissions.unit_id', 'poe_submissions.class_id', 'poe_submissions.status', 'poe_submissions.submitted_at', 'poe_submissions.created_at')
            ->whereIn('status', ['submitted', 'under_review'])
            ->latest('submitted_at')
            ->limit(10)
            ->get()
            ->map(function($submission) {
                // Calculate days since submission (for overdue calculation)
                if ($submission->submitted_at) {
                    $daysSinceSubmission = now()->diffInDays($submission->submitted_at);
                    $submission->days_since_submission = $daysSinceSubmission;
                    $submission->is_overdue = $daysSinceSubmission > 3; // Consider overdue after 3 days
                } else {
                    $submission->days_since_submission = null;
                    $submission->is_overdue = false;
                }
                return $submission;
            });

        // Recent submissions - optimized with select
        $recentSubmissions = (clone $submissionsQuery)
            ->select('poe_submissions.id', 'poe_submissions.student_id', 'poe_submissions.unit_id', 'poe_submissions.class_id', 'poe_submissions.status', 'poe_submissions.submitted_at', 'poe_submissions.created_at')
            ->latest('created_at')
            ->limit(10)
            ->get();

        // Calculate statistics with caching
        $stats = cache()->remember('trainer.dashboard.stats.' . $trainer->id . '.' . $activeTerm->id, 300, function() use ($classIds, $unitIds, $activeTerm, $trainer, $allClassIds) {
            $baseQuery = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $classIds)
                ->whereIn('unit_id', $unitIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                });
            
            // Assignment statistics - filtered by active term AND trainer's assigned classes
            $assignmentsQuery = \App\Models\Assignment::where('trainer_id', $trainer->id)
                ->whereIn('unit_id', $unitIds)
                ->whereIn('class_id', $allClassIds) // CRITICAL: Only assignments for trainer's assigned classes
                ->where('created_at', '>=', $activeTerm->start_date)
                ->where('created_at', '<=', $activeTerm->end_date ?? now());
            
            $assignmentSubmissionsQuery = \App\Models\AssignmentSubmission::whereHas('assignment', function($q) use ($trainer) {
                $q->where('trainer_id', $trainer->id);
            });
            
            return [
                'total_submissions' => (clone $baseQuery)->count(),
                'pending_submissions' => (clone $baseQuery)
                    ->whereIn('status', ['submitted', 'under_review'])
                    ->count(),
                'approved_submissions' => (clone $baseQuery)->where('status', 'validator_approved')->count(),
                'under_review' => (clone $baseQuery)->whereIn('status', ['submitted', 'under_review', 'trainer_reviewed'])->count(),
                'rejected' => (clone $baseQuery)->where('status', 'rejected')->count(),
                // Assignment statistics
                'total_assignments' => (clone $assignmentsQuery)->count(),
                'published_assignments' => (clone $assignmentsQuery)->where('is_published', true)->count(),
                'total_assignment_submissions' => (clone $assignmentSubmissionsQuery)->count(),
                'marked_submissions' => (clone $assignmentSubmissionsQuery)->whereNotNull('marks')->count(),
                'average_assignment_marks' => (clone $assignmentSubmissionsQuery)->whereNotNull('marks')->avg('marks') ?? 0,
            ];
        });
        
        // Student performance overview - optimized with select
        // Wrap in collect() to ensure it's always a collection, even when retrieved from cache
        $studentPerformance = collect(cache()->remember('trainer.dashboard.performance.' . $trainer->id . '.' . $activeTerm->id, 300, function() use ($classIds, $unitIds, $activeTerm) {
            $students = User::select('id', 'name', 'email')
                ->where('role', 'student')
                ->whereHas('enrollments', function($query) use ($classIds) {
                    $query->whereIn('class_id', $classIds);
                })
                ->limit(10)
                ->get();

            return $students->map(function($student) use ($classIds, $unitIds, $activeTerm) {
                // Get submissions without global scope to avoid JOIN conflicts - optimized
                $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
                    ->select('id', 'student_id', 'unit_id', 'class_id', 'status')
                    ->where('student_id', $student->id)
                    ->whereIn('class_id', $classIds)
                    ->whereIn('unit_id', $unitIds)
                    ->whereHas('schoolClass', function($q) use ($activeTerm) {
                        $q->where('term_id', $activeTerm->id);
                    })
                    ->get();
                
                $totalMarks = 0; // Marks are now in assignment_submissions, not reviews
                $submissionsWithMarks = 0;
                $averageMarks = 0;

                return [
                    'student' => $student,
                    'total_submissions' => $submissions->count(),
                    'approved_submissions' => $submissions->where('status', 'validator_approved')->count(),
                    'average_marks' => $averageMarks,
                ];
            })->sortByDesc('total_submissions')->values()->toArray();
        }));

        // Status distribution for analytics
        $statusDistribution = cache()->remember('trainer.dashboard.status_dist.' . $trainer->id . '.' . $activeTerm->id, 300, function() use ($classIds, $unitIds, $activeTerm) {
            // Use withoutGlobalScope to avoid JOIN conflicts, then filter by term manually
            return PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $classIds)
                ->whereIn('unit_id', $unitIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                })
                ->selectRaw('status, count(*) as count')
                ->groupBy('status')
                ->pluck('count', 'status')
                ->toArray();
        });

        // Calculate overdue submissions count
        $overdueSubmissionsCount = $pendingSubmissions->filter(function($submission) {
            return $submission->is_overdue ?? false;
        })->count();

        return view('trainer.dashboard', compact(
            'activeTerm',
            'assignedUnits',
            'stats',
            'pendingSubmissions',
            'recentSubmissions',
            'studentPerformance',
            'totalStudents',
            'statusDistribution',
            'overdueSubmissionsCount',
            'unitIds',
            'classIds'
        ));
    }
}

