<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\Message;
use Illuminate\Support\Facades\Auth;
use Illuminate\Notifications\DatabaseNotification;

class MessageController extends Controller
{
    /**
     * Display a listing of notifications and messages.
     */
    public function index()
    {
        $user = Auth::user();

        // Get Laravel notifications (unread first, then by date)
        $notifications = $user->notifications()
            ->orderBy('read_at', 'asc')
            ->orderBy('created_at', 'desc')
            ->paginate(15, ['*'], 'notifications_page');

        // Get direct messages (unread first, then by date)
        $messages = Message::where('receiver_id', $user->id)
            ->with(['sender:id,name,email,role'])
            ->orderBy('is_read', 'asc')
            ->orderBy('created_at', 'desc')
            ->paginate(15, ['*'], 'messages_page');

        // Count unread items
        $unreadNotifications = $user->unreadNotifications()->count();
        $unreadMessages = Message::where('receiver_id', $user->id)
            ->where('is_read', false)
            ->count();

        return view('trainer.messages.index', compact(
            'notifications',
            'messages',
            'unreadNotifications',
            'unreadMessages'
        ));
    }

    /**
     * Mark a notification as read.
     */
    public function markNotificationAsRead($notificationId)
    {
        $user = Auth::user();
        $notification = DatabaseNotification::find($notificationId);
        
        if ($notification && $notification->notifiable_id === $user->id) {
            $notification->markAsRead();
            return redirect()->back()->with('success', 'Notification marked as read.');
        }
        
        return redirect()->back()->with('error', 'Notification not found.');
    }

    /**
     * Mark a message as read.
     */
    public function markMessageAsRead(Message $message)
    {
        if ($message->receiver_id !== Auth::id()) {
            abort(403);
        }

        $message->update(['is_read' => true]);
        return redirect()->back()->with('success', 'Message marked as read.');
    }

    /**
     * Show a notification.
     */
    public function showNotification($notificationId)
    {
        $user = Auth::user();
        $notification = DatabaseNotification::find($notificationId);
        
        if (!$notification || $notification->notifiable_id !== $user->id) {
            abort(404);
        }

        $notification->markAsRead();

        return view('trainer.messages.show-notification', compact('notification'));
    }

    /**
     * Show a message.
     */
    public function showMessage(Message $message)
    {
        if ($message->receiver_id !== Auth::id()) {
            abort(403);
        }

        $message->update(['is_read' => true]);
        $message->load('sender:id,name,email,role');

        return view('trainer.messages.show-message', compact('message'));
    }
}
