<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\SchoolClass;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class NotificationController extends Controller
{
    /**
     * Display the notification creation form.
     */
    public function index()
    {
        $trainer = Auth::user();
        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return redirect()->route('trainer.dashboard')
                ->with('warning', 'No active term found.');
        }

        // Get classes assigned to this trainer via trainer_unit_class
        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->distinct()
            ->pluck('class_id')
            ->toArray();

        // Get classes for active term
        $classes = SchoolClass::whereIn('id', $classIds)
            ->where('term_id', $activeTerm->id)
            ->select('id', 'name', 'code')
            ->orderBy('name')
            ->get();

        // Get students enrolled in trainer's classes
        $studentIds = DB::table('enrollments')
            ->whereIn('class_id', $classIds)
            ->where('status', 'active')
            ->distinct()
            ->pluck('student_id')
            ->toArray();

        $students = User::whereIn('id', $studentIds)
            ->where('role', 'student')
            ->where('status', 'active')
            ->select('id', 'name', 'email', 'admission_number')
            ->orderBy('name')
            ->get();

        return view('trainer.notifications.index', compact('classes', 'students', 'activeTerm'));
    }

    /**
     * Send individual notification to a student.
     */
    public function sendIndividual(Request $request)
    {
        $trainer = Auth::user();
        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();

        $request->validate([
            'student_id' => ['required', 'exists:users,id'],
            'title' => ['required', 'string', 'max:255'],
            'message' => ['required', 'string'],
        ]);

        $student = User::findOrFail($request->student_id);

        // Verify student is in trainer's classes
        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->distinct()
            ->pluck('class_id')
            ->toArray();

        $isStudentInClass = DB::table('enrollments')
            ->where('student_id', $student->id)
            ->whereIn('class_id', $classIds)
            ->where('status', 'active')
            ->exists();

        if (!$isStudentInClass) {
            return redirect()->back()
                ->with('error', 'You can only send notifications to students in your assigned classes.')
                ->withInput();
        }

        // Verify student role
        if ($student->role !== 'student') {
            return redirect()->back()
                ->with('error', 'You can only send notifications to students.')
                ->withInput();
        }

        // Create notification
        $student->notify(new \App\Notifications\GeneralNotification(
            $request->title,
            $request->message,
            $trainer->id
        ));

        return redirect()->route('trainer.notifications.index')
            ->with('success', 'Notification sent successfully.');
    }

    /**
     * Send bulk notification to a class.
     */
    public function sendToClass(Request $request)
    {
        $trainer = Auth::user();
        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();

        $request->validate([
            'class_id' => ['required', 'exists:classes,id'],
            'title' => ['required', 'string', 'max:255'],
            'message' => ['required', 'string'],
        ]);

        $class = SchoolClass::findOrFail($request->class_id);

        // Verify trainer is assigned to this class
        $isTrainerAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('class_id', $class->id)
            ->exists();

        if (!$isTrainerAssigned) {
            return redirect()->back()
                ->with('error', 'You are not assigned to this class.')
                ->withInput();
        }

        // Get students enrolled in this class
        $studentIds = DB::table('enrollments')
            ->where('class_id', $class->id)
            ->where('status', 'active')
            ->pluck('student_id')
            ->toArray();

        $students = User::whereIn('id', $studentIds)
            ->where('role', 'student')
            ->where('status', 'active')
            ->get();

        if ($students->isEmpty()) {
            return redirect()->back()
                ->with('error', 'No active students found in this class.')
                ->withInput();
        }

        // Send notification to all students
        foreach ($students as $student) {
            $student->notify(new \App\Notifications\GeneralNotification(
                $request->title,
                $request->message,
                $trainer->id
            ));
        }

        return redirect()->route('trainer.notifications.index')
            ->with('success', "Notification sent to {$students->count()} student(s) in class {$class->name}.");
    }

    /**
     * Send bulk notification to all students in trainer's classes.
     */
    public function sendToAll(Request $request)
    {
        $trainer = Auth::user();
        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();

        $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'message' => ['required', 'string'],
        ]);

        // Get classes assigned to this trainer
        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->distinct()
            ->pluck('class_id')
            ->toArray();

        if (empty($classIds)) {
            return redirect()->back()
                ->with('error', 'You are not assigned to any classes.')
                ->withInput();
        }

        // Get all students enrolled in trainer's classes
        $studentIds = DB::table('enrollments')
            ->whereIn('class_id', $classIds)
            ->where('status', 'active')
            ->distinct()
            ->pluck('student_id')
            ->toArray();

        $students = User::whereIn('id', $studentIds)
            ->where('role', 'student')
            ->where('status', 'active')
            ->get();

        if ($students->isEmpty()) {
            return redirect()->back()
                ->with('error', 'No active students found in your assigned classes.')
                ->withInput();
        }

        // Send notification to all students
        foreach ($students as $student) {
            $student->notify(new \App\Notifications\GeneralNotification(
                $request->title,
                $request->message,
                $trainer->id
            ));
        }

        return redirect()->route('trainer.notifications.index')
            ->with('success', "Notification sent to {$students->count()} student(s) in all your assigned classes.");
    }
}
