<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\PracticalEvidence;
use App\Models\PracticalEvidenceTask;
use App\Models\PracticalEvidenceComment;
use App\Models\Unit;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class PortfolioEvidenceController extends Controller
{
    /**
     * Display a listing of portfolio evidence tasks for trainer's units.
     */
    public function index(Request $request)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return view('trainer.portfolio-evidence.index', [
                'tasks' => collect(),
                'units' => collect(),
                'stats' => [
                    'total' => 0,
                    'submitted' => 0,
                    'under_review' => 0,
                    'accepted' => 0,
                    'rejected' => 0,
                    'needs_correction' => 0,
                ],
            ])->with('warning', 'No active term found.');
        }

        // Get active term classes
        $activeTermClassIds = \App\Models\SchoolClass::where('term_id', $activeTerm->id)
            ->pluck('id')
            ->toArray();

        // Get units assigned to this trainer via trainer_unit_class - ONLY for active term classes
        $unitIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->whereIn('class_id', $activeTermClassIds)
            ->distinct()
            ->pluck('unit_id')
            ->toArray();

        // Get class IDs where trainer is assigned (for filtering tasks)
        $assignedClassIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->whereIn('class_id', $activeTermClassIds)
            ->pluck('class_id')
            ->toArray();

        // If no units assigned in active term, return empty
        if (empty($unitIds) || empty($assignedClassIds)) {
            return view('trainer.portfolio-evidence.index', [
                'tasks' => collect(),
                'units' => collect(),
                'stats' => [
                    'total' => 0,
                    'submitted' => 0,
                    'under_review' => 0,
                    'accepted' => 0,
                    'rejected' => 0,
                    'needs_correction' => 0,
                ],
            ])->with('info', 'You don\'t have any units assigned for the current term.');
        }

        // Sync tasks from existing practical evidence (if any)
        $this->syncTasksFromEvidence($unitIds, $assignedClassIds, $activeTerm->id);

        // Get all tasks for trainer's units AND classes
        $query = PracticalEvidenceTask::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $assignedClassIds)
            ->where('term_id', $activeTerm->id)
            ->with(['student:id,name,email', 'unit:id,name,code', 'schoolClass:id,name', 'reviewer:id,name']);

        // Filter by unit
        if ($request->unit) {
            $query->where('unit_id', $request->unit);
        }

        // Filter by status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter by student (search)
        if ($request->search) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }

        $tasks = $query->latest('submitted_at')->paginate(20);

        // Get all units for filter dropdown
        $units = Unit::whereIn('id', $unitIds)->orderBy('name')->get();

        // Get statistics (based on tasks, not files)
        $baseQuery = PracticalEvidenceTask::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $assignedClassIds)
            ->where('term_id', $activeTerm->id);
        
        $stats = [
            'total' => (clone $baseQuery)->count(),
            'submitted' => (clone $baseQuery)->where('status', 'submitted')->count(),
            'under_review' => (clone $baseQuery)->where('status', 'under_review')->count(),
            'accepted' => (clone $baseQuery)->where('status', 'accepted')->count(),
            'rejected' => (clone $baseQuery)->where('status', 'rejected')->count(),
            'needs_correction' => (clone $baseQuery)->where('status', 'needs_correction')->count(),
        ];

        return view('trainer.portfolio-evidence.index', compact('tasks', 'units', 'stats', 'activeTerm'));
    }

    /**
     * Display the specified task with all its files.
     */
    public function show(PracticalEvidenceTask $task)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            abort(403, 'No active term found.');
        }

        // Verify trainer is assigned to this unit in the specific class
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $task->unit_id)
            ->where('class_id', $task->class_id)
            ->exists();

        if (!$isAssigned) {
            abort(403, 'You are not assigned to this unit in this class.');
        }

        // Load relationships
        $task->load([
            'student:id,name,email',
            'unit:id,name,code',
            'schoolClass:id,name,code',
            'reviewer:id,name',
        ]);

        // Get all evidence files for this task
        $evidenceFiles = $task->evidenceFiles();

        // Get all comments for files in this task
        $allComments = PracticalEvidenceComment::whereIn('practical_evidence_id', $evidenceFiles->pluck('id'))
            ->with('user:id,name,email')
            ->orderBy('created_at')
            ->get();

        return view('trainer.portfolio-evidence.show', compact('task', 'evidenceFiles', 'allComments'));
    }

    /**
     * Update the status of a task and sync to all files.
     */
    public function updateStatus(Request $request, PracticalEvidenceTask $task)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            abort(403, 'No active term found.');
        }

        // Verify trainer is assigned to this unit in the specific class
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $task->unit_id)
            ->where('class_id', $task->class_id)
            ->exists();

        if (!$isAssigned) {
            abort(403, 'You are not assigned to this unit in this class.');
        }

        $request->validate([
            'status' => ['required', 'in:under_review,accepted,rejected,needs_correction'],
            'comment' => ['nullable', 'string', 'max:2000'],
        ]);

        DB::beginTransaction();
        try {
            // Update task status
            $task->update([
                'status' => $request->status,
                'trainer_comment' => $request->comment,
                'reviewed_by' => $trainer->id,
                'reviewed_at' => now(),
            ]);

            // Sync status to all files under this task
            // Map task status to file status
            $fileStatus = match($request->status) {
                'under_review' => 'reviewed',
                'accepted' => 'accepted',
                'rejected' => 'rejected',
                'needs_correction' => 'reviewed', // Keep as reviewed for needs correction
                default => 'reviewed',
            };

            // Update all files using query builder
            $task->evidenceFilesQuery()->update(['status' => $fileStatus]);

            // Add comment to all files if provided
            if ($request->comment) {
                $evidenceFiles = $task->evidenceFiles();
                foreach ($evidenceFiles as $evidence) {
                    PracticalEvidenceComment::create([
                        'practical_evidence_id' => $evidence->id,
                        'task_id' => $task->id,
                        'user_id' => $trainer->id,
                        'role' => 'trainer',
                        'comment' => $request->comment,
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('trainer.portfolio-evidence.index')
                ->with('success', 'Task status updated successfully. All files have been updated.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to update task status: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Download an evidence file.
     */
    public function download($evidenceId)
    {
        $trainer = Auth::user();
        $evidence = PracticalEvidence::findOrFail($evidenceId);

        // Verify trainer is assigned to this unit in the specific class
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $evidence->unit_id)
            ->where('class_id', $evidence->class_id)
            ->exists();

        if (!$isAssigned) {
            abort(403, 'You are not assigned to this unit in this class.');
        }

        if (!Storage::disk('public')->exists($evidence->file_path)) {
            return redirect()->back()
                ->with('error', 'File not found.');
        }

        return Storage::disk('public')->download($evidence->file_path, $evidence->original_name);
    }

    /**
     * Sync tasks from existing practical evidence files.
     * This creates task records for existing evidence that doesn't have a task yet.
     */
    private function syncTasksFromEvidence($unitIds, $classIds, $termId)
    {
        // Get all unique task combinations from practical_evidences
        $taskGroups = PracticalEvidence::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds)
            ->where('term_id', $termId)
            ->select('student_id', 'unit_id', 'class_id', 'task_name', 'term_id')
            ->selectRaw('MIN(uploaded_at) as submitted_at')
            ->groupBy('student_id', 'unit_id', 'class_id', 'task_name', 'term_id')
            ->get();

        foreach ($taskGroups as $group) {
            // Check if task already exists
            $existingTask = PracticalEvidenceTask::where('student_id', $group->student_id)
                ->where('unit_id', $group->unit_id)
                ->where('class_id', $group->class_id)
                ->where('task_name', $group->task_name)
                ->where('term_id', $group->term_id)
                ->first();

            if (!$existingTask) {
                // Get the most common status from files (or default to submitted)
                $statusCounts = PracticalEvidence::where('student_id', $group->student_id)
                    ->where('unit_id', $group->unit_id)
                    ->where('class_id', $group->class_id)
                    ->where('task_name', $group->task_name)
                    ->where('term_id', $group->term_id)
                    ->selectRaw('status, COUNT(*) as count')
                    ->groupBy('status')
                    ->orderByDesc('count')
                    ->first();

                $taskStatus = 'submitted';
                if ($statusCounts) {
                    // Map file status to task status
                    $taskStatus = match($statusCounts->status) {
                        'accepted' => 'accepted',
                        'rejected' => 'rejected',
                        'reviewed' => 'under_review',
                        default => 'submitted',
                    };
                }

                // Create task
                PracticalEvidenceTask::create([
                    'student_id' => $group->student_id,
                    'unit_id' => $group->unit_id,
                    'class_id' => $group->class_id,
                    'term_id' => $group->term_id,
                    'task_name' => $group->task_name,
                    'status' => $taskStatus,
                    'submitted_at' => $group->submitted_at,
                ]);
            }
        }
    }
}
