<?php

namespace App\Http\Controllers\Validator;

use App\Http\Controllers\Controller;
use App\Models\PracticalEvidence;
use App\Models\PracticalEvidenceComment;
use App\Models\PracticalEvidenceTask;
use App\Models\Unit;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class PracticalEvidenceController extends Controller
{
    /**
     * Display practical evidence for a student in a unit.
     */
    public function show(Unit $unit, User $student)
    {
        $validator = Auth::user();

        // Get all practical evidence for this student and unit
        $evidences = PracticalEvidence::where('unit_id', $unit->id)
            ->where('student_id', $student->id)
            ->with(['comments.user'])
            ->latest('uploaded_at')
            ->get();

        return view('validator.practical-evidence.show', compact('unit', 'student', 'evidences'));
    }

    /**
     * Add a comment to practical evidence.
     */
    public function addComment(Request $request, PracticalEvidence $practicalEvidence)
    {
        $validator = Auth::user();

        $request->validate([
            'comment' => ['required', 'string', 'max:1000'],
        ]);

        // Find or create the task for this evidence
        $task = PracticalEvidenceTask::firstOrCreate(
            [
                'student_id' => $practicalEvidence->student_id,
                'unit_id' => $practicalEvidence->unit_id,
                'class_id' => $practicalEvidence->class_id,
                'task_name' => $practicalEvidence->task_name,
                'term_id' => $practicalEvidence->term_id,
            ],
            [
                'status' => 'submitted',
                'submitted_at' => $practicalEvidence->uploaded_at ?? now(),
            ]
        );

        PracticalEvidenceComment::create([
            'practical_evidence_id' => $practicalEvidence->id,
            'task_id' => $task->id,
            'user_id' => $validator->id,
            'role' => 'verifier',
            'comment' => $request->comment,
        ]);

        return redirect()->back()
            ->with('success', 'Comment added successfully.');
    }

    /**
     * Download practical evidence file.
     */
    public function download(PracticalEvidence $practicalEvidence)
    {
        if (!Storage::exists($practicalEvidence->file_path)) {
            return redirect()->back()
                ->with('error', 'File not found.');
        }

        return Storage::download($practicalEvidence->file_path, $practicalEvidence->original_name);
    }
}
