<?php

namespace App\Http\Controllers\Validator;

use App\Http\Controllers\Controller;
use App\Models\ValidationRequest;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\UnitValidation;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Models\User;
use App\Services\ActiveTermService;
use App\Notifications\ValidationStatusChanged;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use iio\libmergepdf\Merger;

class UnitController extends Controller
{
    /**
     * Display a list of units pending validation.
     */
    public function index(Request $request)
    {
        $validator = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return view('validator.units.index', [
                'validationRequests' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found.');
        }

        $query = ValidationRequest::where('term_id', $activeTerm->id)
            ->with([
                'schoolClass:id,name,code,department_id',
                'schoolClass.department:id,name',
                'unit:id,name,code',
                'submitter:id,name',
                'validator:id,name'
            ]);

        // Filter by status
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        // Filter by class
        if ($request->has('class') && $request->class) {
            $query->where('class_id', $request->class);
        }

        // Filter by unit
        if ($request->has('unit') && $request->unit) {
            $query->where('unit_id', $request->unit);
        }

        $validationRequests = $query->orderBy('submitted_at', 'desc')->paginate(20);

        // Get filter options
        $classes = SchoolClass::where('term_id', $activeTerm->id)
            ->whereIn('id', ValidationRequest::where('term_id', $activeTerm->id)->pluck('class_id')->unique())
            ->orderBy('name')
            ->get(['id', 'name', 'code']);

        $units = Unit::whereIn('id', ValidationRequest::where('term_id', $activeTerm->id)->pluck('unit_id')->unique())
            ->orderBy('name')
            ->get(['id', 'name', 'code']);

        return view('validator.units.index', compact(
            'validationRequests',
            'classes',
            'units',
            'activeTerm'
        ));
    }

    /**
     * Show validation page for a specific validation request.
     */
    public function show(ValidationRequest $validationRequest)
    {
        $validator = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $validationRequest->term_id !== $activeTerm->id) {
            abort(404, 'Validation request not found or inactive term.');
        }

        // Load relationships
        $validationRequest->load([
            'schoolClass:id,name,code',
            'unit:id,name,code,classification',
            'submitter:id,name',
            'unitValidations.student:id,name,admission_number'
        ]);

        // Get all students in the class
        $students = $validationRequest->schoolClass->students()
            ->wherePivot('status', 'active')
            ->select('users.id', 'users.name', 'users.email', 'users.admission_number')
            ->orderBy('users.name')
            ->get();

        // Get all assignments for this unit in this class
        $assignments = Assignment::where('unit_id', $validationRequest->unit_id)
            ->where('class_id', $validationRequest->class_id)
            ->where('is_published', true)
            ->with(['trainer:id,name'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Get marks for each student and assignment
        $marksData = [];
        foreach ($students as $student) {
            $studentMarks = [];
            $totalMarks = 0;
            $maxMarks = 0;
            
            foreach ($assignments as $assignment) {
                $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->where('student_id', $student->id)
                    ->first();
                
                $mark = $submission ? $submission->marks : null;
                $status = $submission ? $submission->marking_status : 'not_marked';
                
                $studentMarks[$assignment->id] = [
                    'marks' => $mark,
                    'status' => $status,
                    'assignment' => $assignment,
                    'submission' => $submission
                ];

                if ($mark !== null && $status !== 'absent') {
                    $totalMarks += $mark;
                    $maxMarks += $assignment->total_marks;
                } elseif ($status === 'absent') {
                    $maxMarks += $assignment->total_marks; // Count absent as assessed but not contributing to total
                }
            }

            // Get or create unit validation for this student
            $unitValidation = UnitValidation::firstOrCreate(
                [
                    'validation_request_id' => $validationRequest->id,
                    'student_id' => $student->id,
                ],
                [
                    'original_marks' => $maxMarks > 0 ? round(($totalMarks / $maxMarks) * 100, 2) : 0,
                    'decision' => 'pending',
                ]
            );

            $marksData[$student->id] = [
                'marks' => $studentMarks,
                'total' => $totalMarks,
                'max' => $maxMarks,
                'percentage' => $maxMarks > 0 ? round(($totalMarks / $maxMarks) * 100, 2) : 0,
                'validation' => $unitValidation,
            ];
        }

        // Count practical evidence tasks per student
        $evidenceCounts = [];
        if (Schema::hasTable('practical_evidence_tasks')) {
            foreach ($students as $student) {
                $count = \App\Models\PracticalEvidenceTask::where('unit_id', $validationRequest->unit_id)
                    ->where('student_id', $student->id)
                    ->where('class_id', $validationRequest->class_id)
                    ->count();
                $evidenceCounts[$student->id] = $count;
            }
        } else {
            foreach ($students as $student) {
                $count = \App\Models\PracticalEvidence::where('unit_id', $validationRequest->unit_id)
                    ->where('student_id', $student->id)
                    ->count();
                $evidenceCounts[$student->id] = $count;
            }
        }

        return view('validator.units.show', compact(
            'validationRequest',
            'students',
            'assignments',
            'marksData',
            'evidenceCounts',
            'activeTerm'
        ));
    }

    /**
     * Unit summary marks preview/print (HTML) for validator.
     * Uses the same view and data structure as the trainer portal so the output is identical.
     */
    public function printMarks(ValidationRequest $validationRequest)
    {
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $validationRequest->term_id !== $activeTerm->id) {
            abort(404, 'Validation request not found or inactive term.');
        }

        $validationRequest->load([
            'schoolClass:id,name,code',
            'unit:id,name,code,department_id',
            'unit.department:id,name',
            'submitter:id,name',
        ]);

        $unit = $validationRequest->unit;
        $unit->load('department:id,name');
        $selectedClass = $validationRequest->schoolClass;
        $trainer = $validationRequest->submitter; // Show who submitted to validation

        $typeToCategoryMap = [
            'assignment' => 'theory',
            'exam'       => 'theory',
            'practical'  => 'practical',
            'project'    => 'project',
        ];

        $allAssignments = Assignment::where('unit_id', $validationRequest->unit_id)
            ->where('class_id', $validationRequest->class_id)
            ->where('is_published', true)
            ->with(['trainer:id,name'])
            ->orderBy('created_at')
            ->get();

        $groupedAssignments = [
            'theory' => [],
            'practical' => [],
            'oral' => [],
            'project' => [],
        ];
        $typeCounts = [
            'assignment' => 0,
            'exam' => 0,
            'practical' => 0,
            'project' => 0,
        ];

        foreach ($allAssignments as $assignment) {
            if (isset($typeCounts[$assignment->type])) {
                $typeCounts[$assignment->type]++;
            }
            $category = $typeToCategoryMap[$assignment->type] ?? null;
            if ($category && isset($groupedAssignments[$category])) {
                $groupedAssignments[$category][] = $assignment;
            }
        }

        $assessmentStructure = $unit->getAssessmentStructure();
        $assignments = [];
        foreach (['theory', 'practical', 'oral', 'project'] as $type) {
            $count = count($groupedAssignments[$type]);
            $requiredCount = $assessmentStructure[$type] ?? 0;
            if ($count > 0 || $requiredCount > 0) {
                foreach ($groupedAssignments[$type] as $index => $assignment) {
                    $number = $index + 1;
                    $prefix = $type === 'theory' ? 'EXAM' : strtoupper($type);
                    $assignment->display_name = $prefix . $number;
                    $assignment->display_type = $type;
                    $assignments[] = $assignment;
                }
            }
        }

        $students = $validationRequest->schoolClass->students()
            ->wherePivot('status', 'active')
            ->select('users.id', 'users.name', 'users.email', 'users.admission_number')
            ->orderBy('users.name')
            ->get();

        $marksData = [];
        $categoryAverages = [];

        foreach ($students as $student) {
            $studentMarks = [];
            $categoryTotals = [
                'theory' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'practical' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'oral' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'project' => ['total' => 0, 'count' => 0, 'max_total' => 0],
            ];

            foreach ($allAssignments as $assignment) {
                $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->where('student_id', $student->id)
                    ->first();

                $marks = $submission ? $submission->marks : null;
                $markingStatus = $submission ? $submission->marking_status : 'not_marked';
                $percentage = null;

                if ($markingStatus === 'absent') {
                    $marks = 0;
                    $percentage = 0;
                    $categoryKey = $typeToCategoryMap[$assignment->type] ?? null;
                    if ($categoryKey && isset($categoryTotals[$categoryKey])) {
                        $categoryTotals[$categoryKey]['total'] += 0;
                        $categoryTotals[$categoryKey]['max_total'] += $assignment->total_marks ?? 0;
                        $categoryTotals[$categoryKey]['count']++;
                    }
                } elseif ($marks !== null && ($assignment->total_marks ?? 0) > 0) {
                    $percentage = ($marks / $assignment->total_marks) * 100;
                    $categoryKey = $typeToCategoryMap[$assignment->type] ?? null;
                    if ($categoryKey && isset($categoryTotals[$categoryKey])) {
                        $categoryTotals[$categoryKey]['total'] += $marks;
                        $categoryTotals[$categoryKey]['max_total'] += $assignment->total_marks;
                        $categoryTotals[$categoryKey]['count']++;
                    }
                }

                $studentMarks[$assignment->id] = [
                    'marks' => $marks,
                    'percentage' => $percentage,
                    'status' => $submission && $submission->marking_status && $submission->marking_status !== 'not_marked'
                        ? $submission->marking_status
                        : ($submission && $submission->status === 'graded' ? 'marked' : 'not_marked'),
                    'submission' => $submission,
                    'total_marks' => $assignment->total_marks,
                ];
            }

            $categoryAverages[$student->id] = [];
            foreach (['theory', 'practical', 'oral', 'project'] as $type) {
                $cat = $categoryTotals[$type];
                if ($cat['count'] > 0 && $cat['max_total'] > 0) {
                    $categoryAverages[$student->id][$type] = [
                        'average_percentage' => ($cat['total'] / $cat['max_total']) * 100,
                        'count' => $cat['count'],
                        'total_marks' => $cat['total'],
                        'max_total' => $cat['max_total'],
                    ];
                } else {
                    $categoryAverages[$student->id][$type] = [
                        'average_percentage' => null,
                        'count' => 0,
                        'total_marks' => 0,
                        'max_total' => 0,
                    ];
                }
            }
            $marksData[$student->id] = $studentMarks;
        }

        $practicalAssignments = $allAssignments->where('type', 'practical')->values();
        $practicalTestCount = $practicalAssignments->count();
        $practicalUnitPercents = [];
        $practicalUnitAverages = [];

        if ($practicalTestCount > 0 && $students->count() > 0) {
            $studentIds = $students->pluck('id');
            foreach ($students as $student) {
                $practicalUnitPercents[$student->id] = [];
            }

            foreach ($practicalAssignments as $testIndex => $assignment) {
                $structure = \App\Models\AssessorToolStructure::where('assignment_id', $assignment->id)->first();
                $pMax = 0.0;
                $oMax = 0.0;

                if ($structure && is_array($structure->structure ?? null)) {
                    $sections = $structure->structure['sections'] ?? [];
                    foreach ($sections as $row) {
                        $rowId = $row['row_identifier'] ?? '';
                        $marksAvailable = (float) ($row['marks_available'] ?? 0);
                        if (!$rowId || $marksAvailable <= 0) continue;
                        $isPractical = preg_match('/practical/i', $rowId);
                        $isOral = preg_match('/oral/i', $rowId);
                        if ($isPractical) $pMax += $marksAvailable;
                        elseif ($isOral) $oMax += $marksAvailable;
                    }
                }
                if ($pMax <= 0 && $oMax <= 0) {
                    $pMax = (float) ($assignment->total_marks ?: 100);
                }

                $submissions = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->whereIn('student_id', $studentIds)
                    ->get()
                    ->keyBy('student_id');

                $marksBySubmission = collect();
                if ($structure && ($pMax > 0 || $oMax > 0)) {
                    $submissionIds = $submissions->pluck('id')->filter()->values();
                    if ($submissionIds->isNotEmpty()) {
                        $marksBySubmission = \App\Models\AssessorToolMark::where('assignment_id', $assignment->id)
                            ->whereIn('assignment_submission_id', $submissionIds)
                            ->get()
                            ->groupBy('assignment_submission_id');
                    }
                }

                foreach ($students as $student) {
                    $pTotal = 0.0;
                    $oTotal = 0.0;
                    $pPercent = null;
                    $oPercent = null;
                    $submission = $submissions->get($student->id);

                    if ($structure && ($pMax > 0 || $oMax > 0) && $submission && $marksBySubmission->has($submission->id)) {
                        foreach ($marksBySubmission->get($submission->id) as $mark) {
                            $rowId = $mark->row_identifier ?? '';
                            if (preg_match('/practical/i', $rowId)) $pTotal += (float) $mark->marks_obtained;
                            elseif (preg_match('/oral/i', $rowId)) $oTotal += (float) $mark->marks_obtained;
                        }
                        if ($pTotal == 0.0 && $oTotal == 0.0 && $submission->marks !== null && (float) $submission->marks > 0) {
                            $pTotal = (float) $submission->marks;
                        }
                        if ($pMax > 0) $pPercent = ($pTotal / $pMax) * 100;
                        if ($oMax > 0) $oPercent = ($oTotal / $oMax) * 100;
                    } elseif ($submission && $submission->marks !== null && $pMax > 0) {
                        $pTotal = (float) $submission->marks;
                        $pPercent = ($pTotal / $pMax) * 100;
                    }

                    $practicalUnitPercents[$student->id][$testIndex] = ['P' => $pPercent, 'O' => $oPercent];
                }
            }

            foreach ($students as $student) {
                $entries = $practicalUnitPercents[$student->id] ?? [];
                $pValues = [];
                $oValues = [];
                foreach ($entries as $entry) {
                    if (isset($entry['P']) && $entry['P'] !== null) $pValues[] = $entry['P'];
                    if (isset($entry['O']) && $entry['O'] !== null) $oValues[] = $entry['O'];
                }
                $pAvg = count($pValues) > 0 ? array_sum($pValues) / count($pValues) : null;
                $oAvg = count($oValues) > 0 ? array_sum($oValues) / count($oValues) : null;
                $practicalUnitAverages[$student->id] = ['Pav' => $pAvg, 'Oav' => $oAvg];
            }
        }

        $institutionSettings = \App\Models\Setting::where('key', 'like', 'institution.%')
            ->get()
            ->keyBy('key')
            ->map(fn ($s) => $s->value);

        $assessmentPresence = [
            'has_assignment' => $typeCounts['assignment'] > 0,
            'has_exam' => $typeCounts['exam'] > 0,
            'has_practical' => $typeCounts['practical'] > 0,
            'has_project' => $typeCounts['project'] > 0,
        ];

        return view('trainer.marks.print-unit', compact(
            'unit',
            'assignments',
            'students',
            'marksData',
            'activeTerm',
            'groupedAssignments',
            'categoryAverages',
            'assessmentStructure',
            'institutionSettings',
            'trainer',
            'assessmentPresence',
            'practicalUnitPercents',
            'practicalUnitAverages',
            'selectedClass'
        ));
    }

    /**
     * Portfolio view: list students with portfolio evidence for this validation request.
     */
    public function portfolioIndex(ValidationRequest $validationRequest)
    {
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $validationRequest->term_id !== $activeTerm->id) {
            abort(404, 'Validation request not found or inactive term.');
        }

        $validationRequest->load(['schoolClass:id,name,code', 'unit:id,name,code']);

        // Get all active students in the class (full portfolio: every student can have exams/practicals PDFs)
        $students = $validationRequest->schoolClass->students()
            ->wherePivot('status', 'active')
            ->select('users.id', 'users.name', 'users.email', 'users.admission_number')
            ->orderBy('users.name')
            ->get();

        $studentSummaries = [];
        $tasksByStudent = collect();
        $legacyByStudent = collect();

        if (Schema::hasTable('practical_evidence_tasks')) {
            $tasks = \App\Models\PracticalEvidenceTask::where('unit_id', $validationRequest->unit_id)
                ->where('class_id', $validationRequest->class_id)
                ->where('term_id', $validationRequest->term_id)
                ->get();
            $tasksByStudent = $tasks->groupBy('student_id');
        } else {
            $evidences = \App\Models\PracticalEvidence::where('unit_id', $validationRequest->unit_id)
                ->where('class_id', $validationRequest->class_id)
                ->where('term_id', $validationRequest->term_id)
                ->get();
            $legacyByStudent = $evidences->groupBy('student_id');
        }

        foreach ($students as $student) {
            if ($tasksByStudent->isNotEmpty()) {
                $studentTasks = $tasksByStudent->get($student->id, collect());
                $studentSummaries[] = [
                    'student' => $student,
                    'tasks_count' => $studentTasks->count(),
                    'files_count' => $studentTasks->sum(fn ($task) => $task->file_count ?? $task->evidenceFiles()->count()),
                    'last_submitted_at' => optional($studentTasks->max('submitted_at'))->format('M d, Y H:i'),
                ];
            } elseif ($legacyByStudent->isNotEmpty()) {
                $studentEvidences = $legacyByStudent->get($student->id, collect());
                $studentSummaries[] = [
                    'student' => $student,
                    'tasks_count' => $studentEvidences->groupBy('task_name')->count(),
                    'files_count' => $studentEvidences->count(),
                    'last_submitted_at' => optional($studentEvidences->max('uploaded_at'))->format('M d, Y H:i'),
                ];
            } else {
                $studentSummaries[] = [
                    'student' => $student,
                    'tasks_count' => 0,
                    'files_count' => 0,
                    'last_submitted_at' => null,
                ];
            }
        }

        // Sort by student name to keep UI stable
        usort($studentSummaries, function ($a, $b) {
            return strcmp($a['student']->name, $b['student']->name);
        });

        return view('validator.portfolio.index', [
            'validationRequest' => $validationRequest,
            'studentSummaries' => $studentSummaries,
            'activeTerm' => $activeTerm,
        ]);
    }

    /**
     * Portfolio view: gallery of evidence for a specific student.
     */
    public function portfolioStudent(ValidationRequest $validationRequest, User $student)
    {
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $validationRequest->term_id !== $activeTerm->id) {
            abort(404, 'Validation request not found or inactive term.');
        }

        // Ensure student belongs to the class
        $isInClass = $validationRequest->schoolClass
            ->students()
            ->wherePivot('status', 'active')
            ->where('users.id', $student->id)
            ->exists();

        if (!$isInClass) {
            abort(404, 'Student not found in this class.');
        }

        $tasks = collect();
        $legacyEvidences = collect();

        if (Schema::hasTable('practical_evidence_tasks')) {
            $tasks = \App\Models\PracticalEvidenceTask::where('unit_id', $validationRequest->unit_id)
                ->where('class_id', $validationRequest->class_id)
                ->where('term_id', $validationRequest->term_id)
                ->where('student_id', $student->id)
                ->get();
        } else {
            // Fallback to legacy table, group by task for display
            $legacyEvidences = \App\Models\PracticalEvidence::where('unit_id', $validationRequest->unit_id)
                ->where('class_id', $validationRequest->class_id)
                ->where('term_id', $validationRequest->term_id)
                ->where('student_id', $student->id)
                ->orderBy('uploaded_at')
                ->get()
                ->groupBy('task_name');
        }

        return view('validator.portfolio.show', [
            'validationRequest' => $validationRequest,
            'student' => $student,
            'tasks' => $tasks,
            'legacyTaskGroups' => $legacyEvidences,
            'activeTerm' => $activeTerm,
        ]);
    }

    /**
     * Download/view compiled exams PDF for a specific student.
     *
     * We assume compiled PDFs are stored under:
     * storage/app/public/compiled/validation-requests/{validationRequest_id}/students/{student_id}/exams.pdf
     */
    public function portfolioStudentExams(ValidationRequest $validationRequest, User $student)
    {
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $validationRequest->term_id !== $activeTerm->id) {
            abort(404, 'Validation request not found or inactive term.');
        }

        // Ensure student belongs to the class
        $isInClass = $validationRequest->schoolClass
            ->students()
            ->wherePivot('status', 'active')
            ->where('users.id', $student->id)
            ->exists();

        if (!$isInClass) {
            abort(404, 'Student not found in this class.');
        }

        $relativePath = "compiled/validation-requests/{$validationRequest->id}/students/{$student->id}/exams.pdf";

        if (!Storage::disk('public')->exists($relativePath)) {
            // Try to compile a simple exams PDF from existing assignment submission PDFs
            $relativePath = $this->compileExamsPdfIfPossible($validationRequest, $student, $relativePath);
        }

        if (!$relativePath || !Storage::disk('public')->exists($relativePath)) {
            return response()->view('validator.portfolio.pdf-unavailable', [
                'title' => 'Exams document not available',
                'message' => 'There are no exam or theory submissions with PDF files for this student yet. The document will appear here once the student has submitted PDFs for the unit\'s written exams.',
                'backUrl' => route('validator.validation-requests.portfolio', $validationRequest),
            ], 200);
        }

        $downloadName = "Exams_{$student->admission_number}_{$validationRequest->unit->code}.pdf";
        $content = Storage::disk('public')->get($relativePath);

        if (request()->boolean('preview')) {
            return response($content, 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline; filename="' . $downloadName . '"',
            ]);
        }

        return Storage::disk('public')->download($relativePath, $downloadName);
    }

    /**
     * Download/view compiled practicals PDF for a specific student.
     *
     * We assume compiled PDFs are stored under:
     * storage/app/public/compiled/validation-requests/{validationRequest_id}/students/{student_id}/practicals.pdf
     */
    public function portfolioStudentPracticals(ValidationRequest $validationRequest, User $student)
    {
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $validationRequest->term_id !== $activeTerm->id) {
            abort(404, 'Validation request not found or inactive term.');
        }

        // Ensure student belongs to the class
        $isInClass = $validationRequest->schoolClass
            ->students()
            ->wherePivot('status', 'active')
            ->where('users.id', $student->id)
            ->exists();

        if (!$isInClass) {
            abort(404, 'Student not found in this class.');
        }

        $relativePath = "compiled/validation-requests/{$validationRequest->id}/students/{$student->id}/practicals.pdf";

        if (!Storage::disk('public')->exists($relativePath)) {
            // Try to build a simple compiled PDF from existing practical evidence
            $relativePath = $this->compilePracticalsPdfIfPossible($validationRequest, $student, $relativePath);
        }

        if (!$relativePath || !Storage::disk('public')->exists($relativePath)) {
            return response()->view('validator.portfolio.pdf-unavailable', [
                'title' => 'Practicals document not available',
                'message' => 'There are no completed practical assessor tools (with marks) for this student yet. The document will appear here once the trainer has graded the practical assessments and the marked checklists are available.',
                'backUrl' => route('validator.validation-requests.portfolio', $validationRequest),
            ], 200);
        }

        $downloadName = "Practicals_{$student->admission_number}_{$validationRequest->unit->code}.pdf";
        $content = Storage::disk('public')->get($relativePath);

        if (request()->boolean('preview')) {
            return response($content, 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline; filename="' . $downloadName . '"',
            ]);
        }

        return Storage::disk('public')->download($relativePath, $downloadName);
    }

    /**
     * Compile Exams PDF for a student by merging up to three exam PDFs.
     *
     * This looks at non-practical assignments (exam/theory) for the unit + class
     * and compiles one PDF per exam (where available) into a single file so that
     * the validator sees a consolidated Exams PDF in the portfolio.
     */
    protected function compileExamsPdfIfPossible(ValidationRequest $validationRequest, User $student, string $targetRelativePath): ?string
    {
        // Find non-practical assignments (exams / theory) for this unit & class
        $examAssignments = Assignment::where('unit_id', $validationRequest->unit_id)
            ->where('class_id', $validationRequest->class_id)
            ->where('is_published', true)
            ->where('type', '!=', 'practical')
            ->orderBy('start_date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        if ($examAssignments->isEmpty()) {
            return null;
        }

        $pdfPaths = [];

        // Collect at most three distinct exam PDFs for this student
        foreach ($examAssignments as $assignment) {
            if (count($pdfPaths) >= 3) {
                break;
            }

            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();

            if (!$submission || !is_array($submission->attachments)) {
                continue;
            }

            foreach ($submission->attachments as $path) {
                if (Str::endsWith(strtolower($path), '.pdf') && Storage::disk('public')->exists($path)) {
                    // Use one PDF per exam assignment
                    $pdfPaths[] = $path;
                    break;
                }
            }
        }

        if (empty($pdfPaths)) {
            return null;
        }

        Storage::disk('public')->makeDirectory(dirname($targetRelativePath));

        // If only one PDF is available, just copy it directly
        if (count($pdfPaths) === 1) {
            $source = $pdfPaths[0];
            Storage::disk('public')->put($targetRelativePath, Storage::disk('public')->get($source));
            return $targetRelativePath;
        }

        // Merge multiple exam PDFs into a single file for the validator
        try {
            $merger = new Merger();

            foreach ($pdfPaths as $path) {
                $absolutePath = Storage::disk('public')->path($path);
                if (is_readable($absolutePath)) {
                    $merger->addFile($absolutePath);
                }
            }

            $mergedContent = $merger->merge();
            if (!$mergedContent) {
                return null;
            }

            Storage::disk('public')->put($targetRelativePath, $mergedContent);
        } catch (\Throwable $e) {
            \Log::error('Failed to merge exam PDFs for validator portfolio', [
                'validation_request_id' => $validationRequest->id,
                'student_id' => $student->id,
                'message' => $e->getMessage(),
            ]);
            return null;
        }

        return $targetRelativePath;
    }

    /**
     * Compile Practicals PDF for a student by merging completed assessor tool PDFs.
     *
     * We no longer pull raw practical evidence files here. Instead, for each
     * practical assignment in this unit + class we use the generated
     * `completed_assessor_tool_path` (which already includes marks).
     */
    protected function compilePracticalsPdfIfPossible(ValidationRequest $validationRequest, User $student, string $targetRelativePath): ?string
    {
        // Find practical assignments for this unit & class
        $practicalAssignments = Assignment::where('unit_id', $validationRequest->unit_id)
            ->where('class_id', $validationRequest->class_id)
            ->where('is_published', true)
            ->where('type', 'practical')
            ->orderBy('start_date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        if ($practicalAssignments->isEmpty()) {
            return null;
        }

        $pdfPaths = [];

        // For each practical assignment, use the completed assessor tool PDF (with marks)
        foreach ($practicalAssignments as $assignment) {
            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();

            if (!$submission || !$submission->completed_assessor_tool_path) {
                continue;
            }

            $path = $submission->completed_assessor_tool_path;

            if (Str::endsWith(strtolower($path), '.pdf') && Storage::disk('public')->exists($path)) {
                $pdfPaths[] = $path;
            }
        }

        if (empty($pdfPaths)) {
            return null;
        }

        Storage::disk('public')->makeDirectory(dirname($targetRelativePath));

        // If only a single assessor tool exists, copy it directly
        if (count($pdfPaths) === 1) {
            $source = $pdfPaths[0];
            Storage::disk('public')->put($targetRelativePath, Storage::disk('public')->get($source));
            return $targetRelativePath;
        }

        // Merge multiple assessor tool PDFs into a single Practicals PDF
        try {
            $merger = new Merger();

            foreach ($pdfPaths as $path) {
                $absolutePath = Storage::disk('public')->path($path);
                if (is_readable($absolutePath)) {
                    $merger->addFile($absolutePath);
                }
            }

            $mergedContent = $merger->merge();
            if (!$mergedContent) {
                return null;
            }

            Storage::disk('public')->put($targetRelativePath, $mergedContent);
        } catch (\Throwable $e) {
            \Log::error('Failed to merge practical assessor tool PDFs for validator portfolio', [
                'validation_request_id' => $validationRequest->id,
                'student_id' => $student->id,
                'message' => $e->getMessage(),
            ]);
            return null;
        }

        return $targetRelativePath;
    }

    /**
     * Update validation decision for a student.
     */
    public function updateStudentDecision(Request $request, ValidationRequest $validationRequest)
    {
        $validator = Auth::user();
        
        $request->validate([
            'student_id' => 'required|exists:users,id',
            'decision' => 'required|in:approved,modified,rejected',
            'adjusted_marks' => 'nullable|numeric|min:0|max:100',
            'comment' => 'nullable|string|max:1000',
        ]);

        $unitValidation = UnitValidation::where('validation_request_id', $validationRequest->id)
            ->where('student_id', $request->student_id)
            ->firstOrFail();

        $unitValidation->update([
            'decision' => $request->decision,
            'adjusted_marks' => $request->adjusted_marks,
            'validator_comment' => $request->comment,
            'validated_by' => $validator->id,
            'validated_at' => now(),
        ]);

        return back()->with('success', 'Student validation decision updated successfully.');
    }

    /**
     * Finalize unit validation (approve or reject entire unit).
     */
    public function finalize(Request $request, ValidationRequest $validationRequest)
    {
        $validator = Auth::user();
        
        $request->validate([
            'status' => 'required|in:approved,rejected,returned_to_trainer',
            'validation_comment' => 'nullable|string|max:5000',
        ]);

        // Check if all students have been validated
        $pendingValidations = UnitValidation::where('validation_request_id', $validationRequest->id)
            ->where('decision', 'pending')
            ->count();

        if ($pendingValidations > 0 && $request->status === 'approved') {
            return back()->with('error', 'Cannot approve unit: Some students still have pending validation decisions.');
        }

        // Reload relationships before updating
        $validationRequest->load(['schoolClass:id,name,code', 'unit:id,name,code', 'submitter:id,name']);
        
        $validationRequest->update([
            'status' => $request->status,
            'validated_by' => $validator->id,
            'validated_at' => now(),
            'validation_comment' => $request->validation_comment,
        ]);

        // Reload after update to get latest data
        $validationRequest->refresh();
        $validationRequest->load(['schoolClass:id,name,code', 'unit:id,name,code']);

        // Send notification to the trainer who submitted the unit
        try {
            $trainer = User::find($validationRequest->submitted_by);
            if ($trainer) {
                $trainer->notify(new ValidationStatusChanged(
                    $validationRequest,
                    $request->status,
                    $validationRequest->unit->name,
                    $validationRequest->schoolClass->name,
                    $validator->name,
                    $request->validation_comment
                ));
            }
        } catch (\Exception $e) {
            // Log error but don't fail the request
            \Log::error('Failed to send validation notification to trainer: ' . $e->getMessage());
        }

        $statusMessage = match($request->status) {
            'approved' => 'Unit has been approved successfully. Trainer has been notified.',
            'rejected' => 'Unit has been rejected. Trainer has been notified.',
            'returned_to_trainer' => 'Unit has been returned to trainer for corrections. Trainer has been notified.',
            default => 'Unit validation updated.',
        };

        return back()->with('success', $statusMessage);
    }

    /**
     * Get evidence files for a student in a validation request.
     */
    public function getStudentEvidence(ValidationRequest $validationRequest, $studentId)
    {
        $validator = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $validationRequest->term_id !== $activeTerm->id) {
            return response()->json(['error' => 'Invalid or inactive term.'], 404);
        }

        // Verify student is in the class
        $student = $validationRequest->schoolClass->students()
            ->wherePivot('status', 'active')
            ->where('users.id', $studentId)
            ->first();

        if (!$student) {
            return response()->json(['error' => 'Student not found in this class.'], 404);
        }

        $evidenceFiles = [];
        
        // Get practical evidence tasks/files for this student and unit
        if (Schema::hasTable('practical_evidence_tasks')) {
            $tasks = \App\Models\PracticalEvidenceTask::where('unit_id', $validationRequest->unit_id)
                ->where('student_id', $studentId)
                ->where('class_id', $validationRequest->class_id)
                ->with('files')
                ->get();

            foreach ($tasks as $task) {
                foreach ($task->files as $file) {
                    $evidenceFiles[] = [
                        'id' => $file->id,
                        'task_name' => $task->task_name,
                        'file_name' => $file->original_name,
                        'file_path' => $file->file_path,
                        'file_type' => $file->evidence_type,
                        'mime_type' => $file->mime_type,
                        'file_size' => $file->file_size,
                        'uploaded_at' => $file->uploaded_at ? $file->uploaded_at->format('M d, Y H:i') : null,
                    ];
                }
            }
        } else {
            // Fallback to old practical_evidences table
            $evidences = \App\Models\PracticalEvidence::where('unit_id', $validationRequest->unit_id)
                ->where('student_id', $studentId)
                ->get();

            foreach ($evidences as $evidence) {
                $evidenceFiles[] = [
                    'id' => $evidence->id,
                    'task_name' => $evidence->task_name,
                    'file_name' => $evidence->original_name,
                    'file_path' => $evidence->file_path,
                    'file_type' => $evidence->evidence_type,
                    'mime_type' => $evidence->mime_type,
                    'file_size' => $evidence->file_size,
                    'uploaded_at' => $evidence->uploaded_at ? $evidence->uploaded_at->format('M d, Y H:i') : null,
                ];
            }
        }

        return response()->json([
            'student' => [
                'id' => $student->id,
                'name' => $student->name,
                'admission_number' => $student->admission_number,
            ],
            'files' => $evidenceFiles,
        ]);
    }

    /**
     * Download an evidence file.
     */
    public function downloadEvidence($fileId)
    {
        $validator = Auth::user();
        
        // Try new table first
        if (Schema::hasTable('practical_evidence_files')) {
            $file = \App\Models\PracticalEvidenceFile::findOrFail($fileId);
            $filePath = $file->file_path;
            $fileName = $file->original_name;
        } else {
            // Fallback to old table
            $file = \App\Models\PracticalEvidence::findOrFail($fileId);
            $filePath = $file->file_path;
            $fileName = $file->original_name;
        }

        if (!Storage::disk('public')->exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::disk('public')->download($filePath, $fileName);
    }
}
