<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Assignment extends Model
{
    use HasFactory;

    protected $fillable = [
        'unit_id',
        'trainer_id',
        'class_id',
        'title',
        'description',
        'type',
        'total_marks',
        'due_date',
        'start_date',
        'is_published',
        'instructions',
        'attachments',
        'candidate_tool',
        'assessor_tool',
        'results_released',
        'results_locked',
        'results_released_at',
    ];

    protected $casts = [
        'due_date' => 'datetime',
        'start_date' => 'datetime',
        'is_published' => 'boolean',
        'attachments' => 'array',
        'candidate_tool' => 'array',
        'assessor_tool' => 'array',
        'results_released' => 'boolean',
        'results_locked' => 'boolean',
        'results_released_at' => 'datetime',
    ];

    // Relationships
    public function unit()
    {
        return $this->belongsTo(Unit::class);
    }

    public function trainer()
    {
        return $this->belongsTo(User::class, 'trainer_id');
    }

    public function schoolClass()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    // Helper methods
    public function isOverdue()
    {
        return $this->due_date && $this->due_date->isPast();
    }

    public function isActive()
    {
        if (!$this->is_published) {
            return false;
        }

        $now = now();
        
        if ($this->start_date && $now->lt($this->start_date)) {
            return false;
        }

        return true;
    }

    /**
     * Check if assignment is available for students (start date has passed)
     */
    public function isAvailableForStudents()
    {
        if (!$this->is_published) {
            return false;
        }

        // If no start date, it's available
        if (!$this->start_date) {
            return true;
        }

        // Check if start date has passed or is equal to now (with 1 second tolerance)
        // Use timestamp comparison to avoid timezone issues
        $startTimestamp = $this->start_date->timestamp;
        $nowTimestamp = now()->timestamp;
        
        return $nowTimestamp >= $startTimestamp;
    }

    public function getTypeLabel()
    {
        return ucfirst($this->type);
    }

    public function getStatusBadgeClasses()
    {
        if (!$this->is_published) {
            return 'bg-gray-100 text-gray-800';
        }

        if ($this->isOverdue()) {
            return 'bg-red-100 text-red-800';
        }

        if ($this->isActive()) {
            return 'bg-green-100 text-green-800';
        }

        return 'bg-yellow-100 text-yellow-800';
    }

    public function submissions()
    {
        return $this->hasMany(AssignmentSubmission::class);
    }

    public function assessorToolStructure()
    {
        return $this->hasOne(AssessorToolStructure::class);
    }

    public function studentSubmission($studentId)
    {
        return $this->submissions()->where('student_id', $studentId)->first();
    }

    /**
     * Check if student can view assessor tool
     * Only visible after marks are awarded
     */
    public function canStudentViewAssessorTool($studentId)
    {
        if ($this->type !== 'practical') {
            return false;
        }

        $submission = $this->studentSubmission($studentId);
        
        if (!$submission) {
            return false;
        }
        
        // For practical assignments, check if marks have been awarded
        // Marks can be in the marks field or in assessor_tool_marks
        $hasMarks = false;
        
        // Check if marks field is set
        if ($submission->marks !== null && $submission->marks > 0) {
            $hasMarks = true;
        } 
        // Or check if assessor tool marks exist
        elseif ($submission->assessorToolMarks()->count() > 0) {
            $hasMarks = true;
        }
        
        // Check marking status - for practical, marking_status should be 'marked' or status should be 'graded'
        $isMarked = ($submission->marking_status === 'marked') || ($submission->status === 'graded');
        
        return $hasMarks && $isMarked;
    }

    /**
     * Check if student can view candidate tool
     * Visible from start date
     */
    public function canStudentViewCandidateTool()
    {
        if ($this->type !== 'practical') {
            return true; // Regular attachments are always visible
        }

        return $this->isAvailableForStudents();
    }

    /**
     * Get attachments visible to student
     */
    public function getStudentVisibleAttachments($studentId)
    {
        if ($this->type === 'practical') {
            $visible = [];
            
            // Candidate tool visible from start date
            if ($this->canStudentViewCandidateTool() && $this->candidate_tool) {
                $visible = array_merge($visible, $this->candidate_tool);
            }
            
            // Assessor tool only visible after grading
            if ($this->canStudentViewAssessorTool($studentId) && $this->assessor_tool) {
                $visible = array_merge($visible, $this->assessor_tool);
            }
            
            return $visible;
        }
        
        // For non-practical assignments, return regular attachments
        return $this->attachments ?? [];
    }
}
