<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PracticalEvidence extends Model
{
    use HasFactory;

    protected $table = 'practical_evidences';

    protected $fillable = [
        'student_id',
        'unit_id',
        'class_id',
        'term_id',
        'evidence_type',
        'file_path',
        'file_name',
        'original_name',
        'file_size',
        'mime_type',
        'task_name',
        'notes',
        'status',
        'uploaded_at',
    ];

    protected $casts = [
        'uploaded_at' => 'datetime',
        'file_size' => 'integer',
    ];

    // Relationships
    public function student()
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    public function unit()
    {
        return $this->belongsTo(Unit::class);
    }

    public function schoolClass()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    public function term()
    {
        return $this->belongsTo(Term::class);
    }

    public function comments()
    {
        return $this->hasMany(PracticalEvidenceComment::class);
    }

    // Helper methods
    public function isVideo()
    {
        return $this->evidence_type === 'video';
    }

    public function isImage()
    {
        return $this->evidence_type === 'image';
    }

    public function isAudio()
    {
        return $this->evidence_type === 'audio';
    }

    public function isDocument()
    {
        return $this->evidence_type === 'document';
    }

    public function isAccepted()
    {
        return $this->status === 'accepted';
    }

    public function isRejected()
    {
        return $this->status === 'rejected';
    }

    public function isReviewed()
    {
        return $this->status === 'reviewed';
    }

    public function getFileSizeHumanAttribute()
    {
        $bytes = $this->file_size ?? 0;
        $units = ['B', 'KB', 'MB', 'GB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Get status badge color classes
     */
    public function getStatusBadgeClasses()
    {
        return match($this->status) {
            'submitted' => 'bg-blue-100 text-blue-800',
            'reviewed' => 'bg-yellow-100 text-yellow-800',
            'accepted' => 'bg-green-100 text-green-800',
            'rejected' => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    /**
     * Get human-readable status
     */
    public function getStatusLabel()
    {
        return ucfirst($this->status);
    }
}
