<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\File;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        // Register ActiveTermService as singleton
        $this->app->singleton(\App\Services\ActiveTermService::class, function ($app) {
            return new \App\Services\ActiveTermService();
        });
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        // Ensure storage and cache directories exist (fixes "No such file or directory" on cPanel/deploy)
        $dirs = [
            storage_path('framework/sessions'),
            storage_path('framework/views'),
            storage_path('framework/cache/data'),
            storage_path('logs'),
            base_path('bootstrap/cache'),
        ];
        foreach ($dirs as $dir) {
            if (!File::isDirectory($dir)) {
                try {
                    File::makeDirectory($dir, 0755, true);
                } catch (\Throwable $e) {
                    // Ignore (e.g. no permission on server); user must set permissions manually
                }
            }
        }

        // Log slow queries in development (helps identify Supabase performance issues)
        if (config('app.debug') && config('app.env') !== 'production') {
            \DB::listen(function ($query) {
                // Log queries that take longer than 200ms (Supabase network latency)
                if ($query->time > 200) {
                    \Log::warning('Slow Query Detected', [
                        'sql' => $query->sql,
                        'time' => $query->time . 'ms',
                        'bindings' => $query->bindings,
                    ]);
                }
                
                // Log very slow queries (> 1 second) as errors
                if ($query->time > 1000) {
                    \Log::error('Very Slow Query', [
                        'sql' => $query->sql,
                        'time' => $query->time . 'ms',
                        'bindings' => $query->bindings,
                    ]);
                }
            });
        }
    }
}
