<?php

namespace App\Services;

use App\Models\PoeSubmission;
use App\Models\SchoolClass;
use App\Services\ActiveTermService;
use Illuminate\Database\Eloquent\Builder;

class QueryOptimizer
{
    /**
     * Optimize submission query with term filtering
     */
    public static function submissionsQuery(array $filters = []): Builder
    {
        $activeTermId = ActiveTermService::getActiveTermId();
        
        // Start with base query - remove global scope to avoid JOIN conflicts with whereHas
        $query = PoeSubmission::withoutGlobalScope('activeTerm')->select([
            'poe_submissions.id',
            'poe_submissions.student_id',
            'poe_submissions.unit_id',
            'poe_submissions.class_id',
            'poe_submissions.status',
            'poe_submissions.version',
            'poe_submissions.submitted_at',
            'poe_submissions.created_at',
        ]);

        // Use whereHas for term filtering (optimized with exists subquery)
        // This avoids join conflicts when eager loading relationships
        $termId = $filters['term_id'] ?? $activeTermId;
        if ($termId) {
            $query->whereHas('schoolClass', function($q) use ($termId) {
                $q->where('term_id', $termId);
            });
        }

        // Status filter
        if (isset($filters['status']) && $filters['status']) {
            $query->where('poe_submissions.status', $filters['status']);
        }

        // Department filter
        if (isset($filters['department_id']) && $filters['department_id']) {
            $query->whereHas('schoolClass', function($q) use ($filters) {
                $q->where('department_id', $filters['department_id']);
            });
        }

        // Level filter
        if (isset($filters['level_id']) && $filters['level_id']) {
            $query->whereHas('schoolClass', function($q) use ($filters) {
                $q->where('level_id', $filters['level_id']);
            });
        }

        // Class filter
        if (isset($filters['class_id']) && $filters['class_id']) {
            $query->where('poe_submissions.class_id', $filters['class_id']);
        }

        // Unit filter
        if (isset($filters['unit_id']) && $filters['unit_id']) {
            $query->where('poe_submissions.unit_id', $filters['unit_id']);
        }

        // Student filter
        if (isset($filters['student_id']) && $filters['student_id']) {
            $query->where('poe_submissions.student_id', $filters['student_id']);
        }

        // Search filter - use whereHas to avoid join conflicts
        if (isset($filters['search']) && $filters['search']) {
            $search = $filters['search'];
            $query->where(function($q) use ($search) {
                $q->whereHas('student', function($sq) use ($search) {
                    $sq->where('name', 'ilike', "%{$search}%")
                       ->orWhere('email', 'ilike', "%{$search}%");
                })
                ->orWhereHas('unit', function($uq) use ($search) {
                    $uq->where('name', 'ilike', "%{$search}%")
                       ->orWhere('code', 'ilike', "%{$search}%");
                });
            });
        }

        // Eager load relationships with selective columns
        return $query->with([
            'student:id,name,email',
            'unit:id,name,code',
            'schoolClass:id,name,term_id,department_id'
        ]);
    }

    /**
     * Get optimized classes query
     */
    public static function classesQuery(array $filters = []): Builder
    {
        $activeTermId = ActiveTermService::getActiveTermId();
        
        $query = SchoolClass::select([
            'classes.id',
            'classes.term_id',
            'classes.department_id',
            'classes.level_id',
            'classes.name',
            'classes.code',
            'classes.created_at',
        ]);

        // Term filter (default to active term)
        if (isset($filters['term_id']) && $filters['term_id']) {
            $query->where('classes.term_id', $filters['term_id']);
        } elseif ($activeTermId) {
            $query->where('classes.term_id', $activeTermId);
        }

        // Department filter
        if (isset($filters['department_id']) && $filters['department_id']) {
            $query->where('classes.department_id', $filters['department_id']);
        }

        // Level filter
        if (isset($filters['level_id']) && $filters['level_id']) {
            $query->where('classes.level_id', $filters['level_id']);
        }

        return $query->with([
            'term:id,name,status',
            'department:id,name',
            'level:id,name'
        ]);
    }

    /**
     * Get cached filter options
     */
    public static function getCachedFilterOptions()
    {
        $activeTerm = ActiveTermService::getActiveTerm();
        
        return cache()->remember('filter_options.' . ($activeTerm->id ?? 'none'), 1800, function() use ($activeTerm) {
            return [
                'terms' => \App\Models\Term::select('id', 'name', 'status', 'start_date')
                    ->orderBy('start_date', 'desc')
                    ->get(),
                'departments' => \App\Models\Department::select('id', 'name')
                    ->orderBy('name')
                    ->get(),
                'levels' => \App\Models\Level::select('id', 'name')
                    ->orderBy('name')
                    ->get(),
                'classes' => SchoolClass::select('id', 'name', 'term_id', 'department_id')
                    ->when($activeTerm, fn($q) => $q->where('term_id', $activeTerm->id))
                    ->orderBy('name')
                    ->get(),
                'units' => \App\Models\Unit::select('id', 'name', 'code', 'department_id')
                    ->orderBy('name')
                    ->get(),
                'students' => \App\Models\User::select('id', 'name', 'email')
                    ->where('role', 'student')
                    ->orderBy('name')
                    ->get(),
            ];
        });
    }
}

