@extends('layouts.trainer')

@section('title', 'Mark Assessor Tool')
@section('page-title', 'Mark Assessor Tool')

@section('content')
<div class="space-y-4 sm:space-y-6" x-data="markingInterface()">
    <!-- Header -->
    <div class="bg-white rounded-xl shadow-md p-4 sm:p-6">
        <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
            <div class="min-w-0 flex-1">
                <h1 class="text-xl sm:text-2xl font-bold text-gray-900 mb-2 break-words">Mark Assessor Tool</h1>
                <p class="text-sm sm:text-base text-gray-600 break-words">Assignment: {{ $assignment->title }}</p>
                <p class="text-xs text-gray-500 mt-1">The assessor tool structure is shared for all students in this assignment</p>
            </div>
            <div class="flex flex-col sm:flex-row gap-2 sm:gap-3">
                <a href="{{ route('trainer.assessor-tool.create-structure', $assignment) }}" 
                   class="inline-flex items-center justify-center px-4 py-2.5 min-h-[44px] border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 touch-manipulation">
                    Edit Structure
                </a>
                <a href="{{ route('trainer.assignments.show', $assignment) }}" 
                   class="inline-flex items-center min-h-[44px] text-blue-600 hover:text-blue-800 font-medium touch-manipulation">
                    ← Back to Assignment
                </a>
            </div>
        </div>
    </div>

    @if($assignment->type === 'practical')
        <!-- Student List View for Practical Assessments -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h2 class="text-lg font-semibold text-gray-900 mb-4">Students List</h2>
            
            <!-- Summary Statistics -->
            <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-5 gap-3 sm:gap-4 mb-4 sm:mb-6">
                <div class="bg-gray-50 rounded-lg p-4">
                    <p class="text-sm text-gray-600">Total Students</p>
                    <p class="text-2xl font-bold text-gray-900">{{ $stats['total'] ?? 0 }}</p>
                </div>
                <div class="bg-blue-50 rounded-lg p-4 border-l-4 border-blue-500">
                    <p class="text-sm text-blue-700">Students Assessed</p>
                    <p class="text-2xl font-bold text-blue-900">{{ $stats['assessed'] ?? 0 }}/{{ $stats['total'] ?? 0 }}</p>
                </div>
                <div class="bg-green-50 rounded-lg p-4 border-l-4 border-green-500">
                    <p class="text-sm text-green-700">Marked</p>
                    <p class="text-2xl font-bold text-green-900">{{ $stats['marked'] ?? 0 }}</p>
                </div>
                <div class="bg-yellow-50 rounded-lg p-4 border-l-4 border-yellow-500">
                    <p class="text-sm text-yellow-700">Not Marked</p>
                    <p class="text-2xl font-bold text-yellow-900">{{ $stats['not_marked'] ?? 0 }}</p>
                </div>
                <div class="bg-red-50 rounded-lg p-4 border-l-4 border-red-500">
                    <p class="text-sm text-red-700">Assessed but Absent (AB)</p>
                    <p class="text-2xl font-bold text-red-900">{{ $stats['absent'] ?? 0 }}</p>
                </div>
            </div>

            @if($submissions->isEmpty())
                <div class="text-center py-8">
                    <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                    </svg>
                    <h3 class="mt-2 text-sm font-medium text-gray-900">No students found</h3>
                    <p class="mt-1 text-sm text-gray-500">No students are enrolled in classes with this unit.</p>
                </div>
            @else
                <!-- Student Table -->
                <div class="overflow-x-auto -mx-4 sm:mx-0" style="-webkit-overflow-scrolling: touch;">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-3 py-2 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Student Name</th>
                                <th class="px-3 py-2 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Adm No</th>
                                <th class="px-3 py-2 sm:px-6 sm:py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-3 py-2 sm:px-6 sm:py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Total</th>
                                <th class="px-3 py-2 sm:px-6 sm:py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            @foreach($submissions as $sub)
                                    @php
                                    $studentId = $sub->student_id ?? $sub->student->id ?? null;
                                    $markingStatus = $sub->marking_status ?? 'not_marked';
                                    if (isset($sub->id) && $sub->id) {
                                        // Real submission - calculate from marks or assessor tool marks
                                        if ($sub->marks) {
                                            $totalMarks = $sub->marks;
                                        } elseif (isset($sub->assessorToolMarks) && $sub->assessorToolMarks->count() > 0) {
                                            $totalMarks = $sub->assessorToolMarks->sum('marks_obtained');
                                        } else {
                                            $totalMarks = 0;
                                        }
                                    } else {
                                        // Dummy object - no marks yet
                                        $totalMarks = 0;
                                    }
                                @endphp
                                <tr class="hover:bg-blue-50 transition-colors cursor-pointer touch-manipulation" 
                                    onclick="openStudentModal({{ $studentId }}, '{{ addslashes($sub->student->name ?? 'Unknown Student') }}', '{{ $sub->student->admission_number ?? 'N/A' }}', '{{ $markingStatus }}')"
                                    style="cursor: pointer;">
                                    <td class="px-3 py-2 sm:px-6 sm:py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900">
                                            {{ $sub->student->name ?? 'Unknown Student' }}
                                        </div>
                                    </td>
                                    <td class="px-3 py-2 sm:px-6 sm:py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-600">
                                            {{ $sub->student->admission_number ?? 'N/A' }}
                                        </div>
                                    </td>
                                    <td class="px-3 py-2 sm:px-6 sm:py-4 whitespace-nowrap text-center">
                                        @if($markingStatus === 'marked')
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                                <svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                                                </svg>
                                                Marked
                                            </span>
                                        @elseif($markingStatus === 'absent')
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                                                <svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"></path>
                                                </svg>
                                                Assessed but Absent (AB)
                                            </span>
                                        @else
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                                <svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd"></path>
                                                </svg>
                                                Not Marked
                                            </span>
                                        @endif
                                    </td>
                                    <td class="px-3 py-2 sm:px-6 sm:py-4 whitespace-nowrap text-center">
                                        @if($markingStatus === 'absent')
                                            <span class="text-sm font-bold text-red-600">AB</span>
                                        @elseif($markingStatus === 'marked')
                                            <span class="text-sm font-medium text-gray-900">{{ number_format($totalMarks, 2) }}</span>
                                        @else
                                            <span class="text-sm text-gray-400">—</span>
                                        @endif
                                    </td>
                                    <td class="px-3 py-2 sm:px-6 sm:py-4 whitespace-nowrap text-center">
                                        <div class="flex flex-wrap items-center justify-center gap-1.5 sm:gap-2">
                                            <button type="button" onclick="event.stopPropagation(); openStudentModal({{ $studentId }}, '{{ $sub->student->name ?? 'Unknown Student' }}', '{{ $sub->student->admission_number ?? 'N/A' }}', '{{ $markingStatus }}')" 
                                                    class="inline-flex items-center px-2.5 py-2 sm:px-3 sm:py-1.5 min-h-[44px] sm:min-h-0 text-xs font-medium text-blue-700 bg-blue-50 rounded-md hover:bg-blue-100 transition-colors touch-manipulation">
                                                @if($markingStatus === 'marked')
                                                    <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
                                                    </svg>
                                                    View
                                                @elseif($markingStatus === 'absent')
                                                    <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"></path>
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                                    </svg>
                                                    Manage
                                                @else
                                                    <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"></path>
                                                    </svg>
                                                    Assess
                                                @endif
                                            </button>
                                            @if($markingStatus === 'marked')
                                                <a href="{{ route('trainer.assessor-tool.marking', $assignment) }}?student_id={{ $studentId }}" 
                                                   onclick="event.stopPropagation();"
                                                   class="inline-flex items-center px-2.5 py-2 sm:px-3 sm:py-1.5 min-h-[44px] sm:min-h-0 text-xs font-medium text-green-700 bg-green-50 rounded-md hover:bg-green-100 transition-colors touch-manipulation">
                                                    <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                                                    </svg>
                                                    Edit
                                                </a>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            @endif
        </div>
    @else
        <!-- Dropdown Selector for Non-Practical Assessments -->
        <div class="bg-white rounded-xl shadow-md p-4 sm:p-6">
            <div class="flex flex-col sm:flex-row sm:items-center gap-3 sm:gap-4">
                <label for="submission-select" class="text-sm font-medium text-gray-700">Select Submission:</label>
                <div class="flex-1 w-full min-w-0 max-w-full sm:max-w-md">
                    <select id="submission-select" 
                            x-model="selectedSubmissionId"
                            @change="loadSubmission()"
                            class="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500">
                        <option value="">-- Select a submission --</option>
                    @forelse($submissions as $sub)
                        @php
                            $studentId = $sub->student_id ?? $sub->student->id ?? null;
                            $isSelected = $submission && ($submission->student_id === $studentId || $submission->id === $sub->id);
                        @endphp
                        <option value="{{ $studentId }}" {{ $isSelected ? 'selected' : '' }}>
                            {{ $sub->student->name ?? 'Unknown Student' }} 
                            @if($sub->student && $sub->student->admission_number)
                                ({{ $sub->student->admission_number }})
                            @endif
                            @if($sub->id)
                                - {{ $sub->created_at ? $sub->created_at->format('M d, Y') : 'N/A' }}
                                @if($sub->marking_status)
                                    - {{ ucfirst(str_replace('_', ' ', $sub->marking_status)) }}
                                @elseif($sub->status)
                                    - {{ ucfirst($sub->status) }}
                                @endif
                            @else
                                - Not Marked
                            @endif
                        </option>
                    @empty
                        <option value="" disabled>No students found for this assignment</option>
                    @endforelse
                    </select>
                    @if($submissions->isEmpty())
                        <p class="mt-2 text-sm text-amber-600">
                            <strong>No students found.</strong> No students are enrolled in classes with this unit.
                        </p>
                    @else
                        <p class="mt-2 text-xs text-gray-500">{{ $submissions->count() }} student(s) available for marking</p>
                    @endif
                </div>
            </div>
        </div>
    @endif

    <!-- Student Action Modal -->
    <div id="studentModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50 p-4">
        <div class="relative top-4 sm:top-20 mx-auto p-4 sm:p-5 border w-full max-w-sm sm:max-w-md shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-medium leading-6 text-gray-900 mb-4">Student Action</h3>
                <div class="mb-4">
                    <p class="text-sm text-gray-600"><strong>Name:</strong> <span id="modalStudentName"></span></p>
                    <p class="text-sm text-gray-600"><strong>Admission No:</strong> <span id="modalStudentAdmNo"></span></p>
                    <p class="text-sm text-gray-600"><strong>Status:</strong> <span id="modalStudentStatus"></span></p>
                </div>
                <div class="flex flex-col gap-3">
                    <button type="button" id="btnAssess" onclick="assessStudent()" 
                            class="w-full px-4 py-3 min-h-[44px] bg-green-600 text-white rounded-md hover:bg-green-700 text-sm font-medium transition-colors flex items-center justify-center gap-2 touch-manipulation">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"></path>
                        </svg>
                        Assess
                    </button>
                    <button type="button" id="btnView" onclick="viewStudent()" 
                            class="hidden w-full px-4 py-3 min-h-[44px] bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium transition-colors flex items-center justify-center gap-2 touch-manipulation">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
                        </svg>
                        View Marks
                    </button>
                    <button type="button" id="btnEdit" onclick="editStudent()" 
                            class="hidden w-full px-4 py-3 min-h-[44px] bg-green-600 text-white rounded-md hover:bg-green-700 text-sm font-medium transition-colors flex items-center justify-center gap-2 touch-manipulation">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                        </svg>
                        Edit Marks
                    </button>
                    <button type="button" id="btnMarkAbsent" onclick="markAbsent()" 
                            class="w-full px-4 py-3 min-h-[44px] bg-red-600 text-white rounded-md hover:bg-red-700 text-sm font-medium transition-colors flex items-center justify-center gap-2 touch-manipulation">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                        Mark as Absent
                    </button>
                    <button type="button" id="btnRemoveAbsent" onclick="removeAbsent()" 
                            class="hidden w-full px-4 py-3 min-h-[44px] bg-yellow-600 text-white rounded-md hover:bg-yellow-700 text-sm font-medium transition-colors flex items-center justify-center gap-2 touch-manipulation">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                        </svg>
                        Remove Absent Status
                    </button>
                    <button type="button" onclick="closeStudentModal()" 
                            class="w-full px-4 py-3 min-h-[44px] bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400 text-sm font-medium transition-colors touch-manipulation">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Marking Form Section (Shown only when a student is selected via query param) -->
    @if($submission && $assignment->type === 'practical')
        @if(!$structure)
            <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-4">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <svg class="h-5 w-5 text-yellow-400" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd" />
                        </svg>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm text-yellow-700">
                            <strong>Assessor tool structure not found.</strong> Please create the structure first.
                        </p>
                        <div class="mt-2">
                            <a href="{{ route('trainer.assessor-tool.create-structure', $assignment) }}" 
                               class="text-sm font-medium text-yellow-800 underline">
                                Create Assessor Tool Structure
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        @else
        @php
            $structureData = $structure->structure ?? [];
            $sections = $structureData['sections'] ?? [];
        @endphp
        
        <div id="markingFormSection" class="bg-white rounded-xl shadow-md p-4 sm:p-6 border-l-4 border-blue-500" style="scroll-margin-top: 20px;">
            <div class="mb-4 sm:mb-6 flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                <div class="min-w-0">
                    <h2 class="text-lg sm:text-xl font-bold text-gray-900 break-words">Marking: {{ $submission->student->name ?? 'Student' }}</h2>
                    <p class="text-sm text-gray-600 mt-1">Admission No: {{ $submission->student->admission_number ?? 'N/A' }}</p>
                </div>
                <div class="flex flex-wrap gap-2 sm:gap-3">
                    <button type="button" onclick="closeAssessmentTool()" 
                            class="inline-flex items-center justify-center px-4 py-2.5 min-h-[44px] text-sm font-medium text-gray-700 bg-gray-100 rounded-md hover:bg-gray-200 transition-colors touch-manipulation">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                        Close Assessment
                    </button>
                    <a href="{{ route('trainer.assessor-tool.marking', $assignment) }}" 
                       class="inline-flex items-center justify-center px-4 py-2.5 min-h-[44px] text-sm font-medium text-white bg-red-600 rounded-md hover:bg-red-700 transition-colors touch-manipulation"
                       title="Close and return to student list">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                        Close & Reload
                    </a>
                </div>
            </div>
        @if(empty($sections))
            <!-- No Structure Items -->
            <div class="bg-white rounded-xl shadow-md p-12 text-center">
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                <h3 class="mt-2 text-sm font-medium text-gray-900">No Items in Assessor Tool Structure</h3>
                <p class="mt-1 text-sm text-gray-500">
                    The assessor tool structure exists but has no items. Please add items to the structure before marking submissions.
                </p>
                <div class="mt-4">
                    <a href="{{ route('trainer.assessor-tool.create-structure', $assignment) }}" 
                       class="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 font-medium">
                        Edit Structure
                    </a>
                </div>
            </div>
        @else
        <!-- Marking Form -->
        <form id="marking-form" @submit.prevent="saveMarks()" class="bg-white rounded-xl shadow-md p-6">
            @csrf
            <input type="hidden" name="submission_id" value="{{ $submission->id }}">

            <!-- Student Info Card -->
            <div class="mb-4 sm:mb-6 p-4 sm:p-5 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg border border-blue-200">
                <div class="grid grid-cols-2 md:grid-cols-4 gap-3 sm:gap-4">
                    <div>
                        <p class="text-xs font-medium text-gray-600 uppercase tracking-wide">Student Name</p>
                        <p class="text-sm font-semibold text-gray-900 mt-1">{{ $submission->student->name }}</p>
                    </div>
                    @if($submission->student->admission_number)
                        <div>
                            <p class="text-xs font-medium text-gray-600 uppercase tracking-wide">Admission Number</p>
                            <p class="text-sm font-semibold text-gray-900 mt-1">{{ $submission->student->admission_number }}</p>
                        </div>
                    @endif
                    <div>
                        <p class="text-xs font-medium text-gray-600 uppercase tracking-wide">Submitted</p>
                        <p class="text-sm font-semibold text-gray-900 mt-1">{{ $submission->created_at->format('M d, Y') }}</p>
                    </div>
                    <div>
                        <p class="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Marks</p>
                        <p class="text-lg font-bold text-blue-600 mt-1" x-text="totalMarks.toFixed(2)">{{ $submission->marks ?? '0.00' }}</p>
                    </div>
                </div>
            </div>

            <!-- Marking Table -->
            <div class="mb-4">
                <h3 class="text-lg font-semibold text-gray-900 mb-3">Evaluation Items</h3>
            </div>
            <div class="overflow-x-auto border border-gray-200 rounded-lg -mx-4 sm:mx-0" style="-webkit-overflow-scrolling: touch;">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-100">
                        <tr>
                            <th class="px-3 py-2 sm:px-6 sm:py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Items to be Evaluated</th>
                            <th class="px-3 py-2 sm:px-6 sm:py-4 text-center text-xs font-bold text-gray-700 uppercase tracking-wider">Marks Available</th>
                            <th class="px-3 py-2 sm:px-6 sm:py-4 text-center text-xs font-bold text-gray-700 uppercase tracking-wider">Marks Obtained</th>
                            <th class="px-3 py-2 sm:px-6 sm:py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Comments</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @php
                            $currentSection = null;
                            $structureData = $structure->structure ?? [];
                            $sections = $structureData['sections'] ?? [];
                        @endphp
                        @if(empty($sections))
                            <tr>
                                <td colspan="4" class="px-4 py-8 text-center text-gray-500">
                                    <p class="mb-2">No items found in the assessor tool structure.</p>
                                    <a href="{{ route('trainer.assessor-tool.create-structure', $assignment) }}" 
                                       class="text-blue-600 hover:text-blue-800 font-medium underline">
                                        Create or Edit Structure
                                    </a>
                                </td>
                            </tr>
                        @else
                            @foreach($sections as $index => $item)
                                @php
                                    $itemSection = $item['section'] ?? null;
                                @endphp
                                @if($itemSection && $itemSection !== $currentSection)
                                    @php $currentSection = $itemSection; @endphp
                                    <tr class="bg-blue-100 border-t-2 border-blue-300">
                                        <td colspan="4" class="px-6 py-3 text-sm font-bold text-blue-900 uppercase tracking-wide">
                                            {{ $currentSection }}
                                        </td>
                                    </tr>
                                @endif
                                <tr class="hover:bg-blue-50 transition-colors">
                                    <td class="px-3 py-2 sm:px-6 sm:py-4">
                                        <div class="text-sm font-medium text-gray-900">{{ $item['item_description'] ?? 'N/A' }}</div>
                                    </td>
                                    <td class="px-3 py-2 sm:px-6 sm:py-4 text-center">
                                        <span class="inline-flex items-center px-2 sm:px-3 py-1 rounded-md text-sm font-semibold text-gray-700 bg-gray-100">
                                            {{ number_format($item['marks_available'] ?? 0, 2) }}
                                        </span>
                                    </td>
                                    <td class="px-3 py-2 sm:px-6 sm:py-4 text-center">
                                        <input type="number" 
                                               name="marks[{{ $item['row_identifier'] ?? 'item_' . $index }}][marks_obtained]"
                                               x-model="marks['{{ $item['row_identifier'] ?? 'item_' . $index }}'].marks_obtained"
                                               @input="updateTotal()"
                                               step="0.01" 
                                               min="0" 
                                               max="{{ $item['marks_available'] ?? 0 }}"
                                               value="{{ isset($marks[$item['row_identifier'] ?? 'item_' . $index]) ? $marks[$item['row_identifier'] ?? 'item_' . $index]->marks_obtained : 0 }}"
                                               class="w-20 sm:w-28 px-2 sm:px-3 py-2 border-2 border-gray-300 rounded-md text-sm font-semibold text-center focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                                               required>
                                    </td>
                                    <td class="px-3 py-2 sm:px-6 sm:py-4">
                                        <textarea name="marks[{{ $item['row_identifier'] ?? 'item_' . $index }}][comments]"
                                                  x-model="marks['{{ $item['row_identifier'] ?? 'item_' . $index }}'].comments"
                                                  rows="2"
                                                  class="w-full text-sm rounded-md border-gray-300 focus:border-blue-500 focus:ring-blue-500"
                                                  placeholder="Add comments (optional)..."
                                                  class="w-full px-3 py-2 border-2 border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors resize-none">{{ isset($marks[$item['row_identifier'] ?? 'item_' . $index]) ? $marks[$item['row_identifier'] ?? 'item_' . $index]->comments : '' }}</textarea>
                                    </td>
                                </tr>
                            @endforeach
                        @endif
                        <!-- Total Row -->
                        <tr class="bg-gray-100 font-semibold">
                            <td class="px-4 py-3 text-sm text-gray-900">TOTAL</td>
                            <td class="px-4 py-3 text-center text-sm text-gray-600">
                                {{ number_format($structure->total_marks ?? 0, 2) }}
                            </td>
                            <td class="px-4 py-3 text-center text-sm text-gray-900" x-text="totalMarks.toFixed(2)">0.00</td>
                            <td class="px-4 py-3"></td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <!-- Total Marks Display and Save Button -->
            <div class="mt-4 sm:mt-6 p-4 sm:p-5 bg-gradient-to-r from-green-50 to-emerald-50 rounded-lg border-2 border-green-200">
                <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                    <div>
                        <p class="text-sm font-medium text-gray-600 uppercase tracking-wide">Total Marks Obtained</p>
                        <p class="text-2xl sm:text-3xl font-bold text-green-600 mt-1" x-text="totalMarks.toFixed(2)">0.00</p>
                        <p class="text-xs text-gray-500 mt-1">out of {{ number_format($assignment->total_marks ?? 0, 2) }} marks</p>
                    </div>
                    <button type="submit" 
                            @click="saveMarks()"
                            :disabled="isSaving"
                            :class="isSaving ? 'opacity-50 cursor-not-allowed' : ''"
                            class="w-full sm:w-auto px-6 sm:px-8 py-3 min-h-[44px] bg-green-600 text-white rounded-lg hover:bg-green-700 font-semibold text-base shadow-lg hover:shadow-xl transition-all flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed touch-manipulation">
                        <svg x-show="!isSaving" class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                        </svg>
                        <svg x-show="isSaving" class="animate-spin h-5 w-5" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        <span x-text="isSaving ? 'Saving...' : 'Save Marks'">Save Marks</span>
                    </button>
                </div>
            </div>
            
            <!-- Additional Actions -->
            <div class="mt-4 flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3">
                <button type="button" 
                        onclick="closeAssessmentTool()"
                        class="inline-flex items-center justify-center px-4 py-2.5 min-h-[44px] text-sm font-medium text-gray-700 bg-gray-100 rounded-md hover:bg-gray-200 transition-colors touch-manipulation">
                    <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                    Close Assessment
                </button>
                <div class="flex items-center gap-3">
                    @if($submission->marks)
                        <a href="{{ route('trainer.assessor-tool.generate', ['assignment' => $assignment, 'submission' => $submission]) }}" 
                           target="_blank"
                           class="px-6 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 font-medium">
                            Generate Document
                        </a>
                    @endif
                </div>
            </div>
        </form>
        @endif
        @endif
    @elseif($assignment->type === 'practical' && !$submission)
        <!-- No student selected for practical assessment -->
        <div class="bg-white rounded-xl shadow-md p-12 text-center">
            <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
            </svg>
            <h3 class="mt-2 text-sm font-medium text-gray-900">No student selected</h3>
            <p class="mt-1 text-sm text-gray-500">Please select a student from the list above to begin marking.</p>
        </div>
    @else
        <div class="bg-white rounded-xl shadow-md p-12 text-center">
            @if($submissions->isEmpty())
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                <h3 class="mt-2 text-sm font-medium text-gray-900">No Submissions Available</h3>
                <p class="mt-1 text-sm text-gray-500">
                    No students have submitted this assignment yet. Submissions will appear here once students submit their work.
                </p>
                <div class="mt-4">
                    <a href="{{ route('trainer.assignments.show', $assignment) }}" 
                       class="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium">
                        Back to Assignment
                    </a>
                </div>
            @else
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                <h3 class="mt-2 text-sm font-medium text-gray-900">No submission selected</h3>
                <p class="mt-1 text-sm text-gray-500">Please select a submission from the dropdown above to begin marking.</p>
            @endif
        </div>
    @endif
</div>

<script>
function markingInterface() {
    return {
        selectedSubmissionId: {{ $submission ? $submission->student_id : (isset($selectedStudentId) && $selectedStudentId ? $selectedStudentId : 'null') }},
        submissionId: {{ $submission && $submission->id ? $submission->id : 'null' }},
        marks: @json($submission ? $marks->mapWithKeys(function($mark) {
            return [$mark->row_identifier => [
                'marks_obtained' => (float) $mark->marks_obtained,
                'comments' => $mark->comments ?? ''
            ]];
        })->toArray() : []),
        totalMarks: {{ $submission ? ($submission->marks ?? 0) : 0 }},
        structure: @json(($structure->structure['sections'] ?? [])),
        isSaving: false,

        init() {
            // Initialize marks for all items
            if (this.structure && Array.isArray(this.structure)) {
                this.structure.forEach(item => {
                    if (!this.marks[item.row_identifier]) {
                        this.marks[item.row_identifier] = {
                            marks_obtained: 0,
                            comments: ''
                        };
                    }
                });
            }
            this.updateTotal();
            
            // Scroll to marking form if student_id is in URL
            const urlParams = new URLSearchParams(window.location.search);
            const studentId = urlParams.get('student_id');
            if (studentId && !isNaN(studentId)) {
                // Wait for DOM to be fully loaded
                setTimeout(() => {
                    const markingForm = document.getElementById('markingFormSection');
                    if (markingForm) {
                        // Scroll to form
                        markingForm.scrollIntoView({ behavior: 'smooth', block: 'start' });
                        
                        // Highlight briefly to draw attention
                        markingForm.style.transition = 'box-shadow 0.3s ease-in-out';
                        markingForm.style.boxShadow = '0 0 0 4px rgba(59, 130, 246, 0.5)';
                        
                        setTimeout(() => {
                            markingForm.style.boxShadow = '';
                        }, 2000);
                    }
                }, 500);
            }
        },

        updateTotal() {
            this.totalMarks = Object.values(this.marks).reduce((sum, mark) => {
                return sum + (parseFloat(mark.marks_obtained) || 0);
            }, 0);
        },

        loadSubmission() {
            // Only used for non-practical assessments
            @if($assignment->type !== 'practical')
            if (this.selectedSubmissionId) {
                window.location.href = '{{ route("trainer.assessor-tool.marking", $assignment) }}?student_id=' + this.selectedSubmissionId;
            } else {
                window.location.href = '{{ route("trainer.assessor-tool.marking", $assignment) }}';
            }
            @endif
        },

        async saveMarks() {
            // Prevent multiple simultaneous saves
            if (this.isSaving) {
                return;
            }

            const form = document.getElementById('marking-form');
            const formData = new FormData(form);
            
            if (!this.submissionId || this.submissionId === null) {
                alert('Please select a submission first.');
                return;
            }
            
            // Set saving state
            this.isSaving = true;
            
            let url;
            @if($submission)
                url = '{{ route("trainer.assessor-tool.save-marks", ["assignment" => $assignment, "submission" => $submission]) }}';
            @else
                url = '{{ route("trainer.assessor-tool.save-marks-new", $assignment) }}';
                formData.append('student_id', this.selectedSubmissionId);
            @endif
            
            try {
                const response = await fetch(url, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || formData.get('_token')
                    }
                });

                const data = await response.json();
                
                if (data.success) {
                    this.totalMarks = parseFloat(data.total_marks);
                    alert('Your edits have been saved and updated in the assessor tool with marks. The student list and totals will refresh.');
                    // Reload so the list and form show updated marks
                    window.location.reload();
                } else {
                    alert('Error: ' + data.message);
                    this.isSaving = false;
                }
            } catch (error) {
                alert('Error saving marks: ' + error.message);
                this.isSaving = false;
            }
        }
    }
}

// Global functions for student modal actions
let currentStudentId = null;
let currentStudentStatus = null;

function openStudentModal(studentId, studentName, admNo, status) {
    currentStudentId = studentId;
    currentStudentStatus = status;
    
    document.getElementById('modalStudentName').textContent = studentName;
    document.getElementById('modalStudentAdmNo').textContent = admNo;
    
    // Update status display
    const statusMap = {
        'marked': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800"><svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path></svg>Marked</span>',
        'absent': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800"><svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"></path></svg>Assessed but Absent (AB)</span>',
        'not_marked': '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800"><svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd"></path></svg>Not Marked</span>'
    };
    document.getElementById('modalStudentStatus').innerHTML = statusMap[status] || statusMap['not_marked'];
    
    // Show/hide buttons based on status
    const btnAssess = document.getElementById('btnAssess');
    const btnView = document.getElementById('btnView');
    const btnEdit = document.getElementById('btnEdit');
    const btnMarkAbsent = document.getElementById('btnMarkAbsent');
    const btnRemoveAbsent = document.getElementById('btnRemoveAbsent');
    
    if (status === 'marked') {
        btnAssess.classList.add('hidden');
        btnView.classList.remove('hidden');
        btnEdit.classList.remove('hidden');
        btnMarkAbsent.classList.add('hidden');
        btnRemoveAbsent.classList.add('hidden');
    } else if (status === 'absent') {
        btnAssess.classList.remove('hidden');
        btnView.classList.add('hidden');
        btnEdit.classList.add('hidden');
        btnMarkAbsent.classList.add('hidden');
        btnRemoveAbsent.classList.remove('hidden');
    } else {
        btnAssess.classList.remove('hidden');
        btnView.classList.add('hidden');
        btnEdit.classList.add('hidden');
        btnMarkAbsent.classList.remove('hidden');
        btnRemoveAbsent.classList.add('hidden');
    }
    
    document.getElementById('studentModal').classList.remove('hidden');
}

function closeStudentModal() {
    document.getElementById('studentModal').classList.add('hidden');
    currentStudentId = null;
    currentStudentStatus = null;
}

function closeAssessmentTool() {
    // Hide the marking form section
    const markingForm = document.getElementById('markingFormSection');
    if (markingForm) {
        // Smooth fade out animation
        markingForm.style.transition = 'opacity 0.3s ease-out, transform 0.3s ease-out';
        markingForm.style.opacity = '0';
        markingForm.style.transform = 'translateY(-10px)';
        
        setTimeout(() => {
            markingForm.style.display = 'none';
            // Remove student_id from URL without reload
            const url = new URL(window.location);
            url.searchParams.delete('student_id');
            window.history.pushState({}, '', url);
            // Scroll to top of student list
            const studentList = document.querySelector('.bg-white.rounded-xl.shadow-md');
            if (studentList) {
                studentList.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }
        }, 300);
    } else {
        // Fallback: reload page without student_id
        const url = new URL(window.location);
        url.searchParams.delete('student_id');
        window.location.href = url.toString();
    }
}

function assessStudent() {
    if (!currentStudentId || currentStudentId === null || currentStudentId === 'null') {
        alert('Invalid student selected. Please try again.');
        return;
    }
    
    const studentId = parseInt(currentStudentId);
    if (isNaN(studentId) || studentId <= 0) {
        alert('Invalid student ID. Please try again.');
        return;
    }
    
    closeStudentModal();
    
    // Navigate to marking page with student_id parameter
    const baseUrl = '{{ route("trainer.assessor-tool.marking", $assignment) }}';
    const url = baseUrl + '?student_id=' + studentId;
    
    window.location.href = url;
}

function viewStudent() {
    assessStudent(); // Same functionality - open marking form
}

function editStudent() {
    assessStudent(); // Same functionality - open marking form for editing
}

function markAbsent() {
    if (!currentStudentId || currentStudentId === null || currentStudentId === 'null') {
        alert('Invalid student selected.');
        return;
    }
    
    const studentId = parseInt(currentStudentId);
    if (isNaN(studentId) || studentId <= 0) {
        alert('Invalid student ID.');
        return;
    }
    
    if (!confirm('Mark this student as absent?')) {
        return;
    }
    
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
    if (!csrfToken) {
        alert('Security token not found. Please refresh the page.');
        return;
    }
    
    fetch('{{ route("trainer.practical-marks.mark-absent", $assignment) }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken,
            'Accept': 'application/json'
        },
        body: JSON.stringify({
            student_id: studentId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            alert('Student marked as assessed but absent (AB).');
            window.location.reload();
        } else {
            alert('Error: ' + (data.message || 'Failed to mark student as absent'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error marking student as absent. Please try again.');
    });
}

function removeAbsent() {
    if (!currentStudentId || currentStudentId === null || currentStudentId === 'null') {
        alert('Invalid student selected.');
        return;
    }
    
    const studentId = parseInt(currentStudentId);
    if (isNaN(studentId) || studentId <= 0) {
        alert('Invalid student ID.');
        return;
    }
    
    if (!confirm('Remove absent status for this student?')) {
        return;
    }
    
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
    if (!csrfToken) {
        alert('Security token not found. Please refresh the page.');
        return;
    }
    
    fetch('{{ route("trainer.practical-marks.remove-absent", $assignment) }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken,
            'Accept': 'application/json'
        },
        body: JSON.stringify({
            student_id: studentId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            alert('Absent status removed.');
            window.location.reload();
        } else {
            alert('Error: ' + (data.message || 'Failed to remove absent status'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error removing absent status. Please try again.');
    });
}
</script>
@endsection

