<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SchoolClass;
use App\Models\Term;
use App\Models\Department;
use App\Models\Level;
use App\Models\Unit;
use App\Services\ActiveTermService;
use App\Services\QueryOptimizer;
use Illuminate\Http\Request;

class ClassController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = SchoolClass::with(['term', 'department', 'level']);

        // Filter by term if provided (non-empty value means specific term selected)
        if ($request->filled('term')) {
            $query->where('term_id', $request->term);
        } else {
            // Default to active term when no filter is selected
            $activeTermId = ActiveTermService::getActiveTermId();
            if ($activeTermId) {
                $query->where('term_id', $activeTermId);
            }
        }

        // Filter by department if provided
        if ($request->has('department') && $request->department) {
            $query->where('department_id', $request->department);
        }

        // Filter by level if provided
        if ($request->has('level') && $request->level) {
            $query->where('level_id', $request->level);
        }

        $classes = $query->withCount(['enrollments', 'units'])->latest()->paginate(15);
        $terms = Term::orderBy('start_date', 'desc')->get();
        $departments = Department::orderBy('name')->get();
        $levels = Level::orderBy('name')->get();
        $activeTerm = ActiveTermService::getActiveTerm();

        return view('admin.classes.index', compact('classes', 'terms', 'departments', 'levels', 'activeTerm'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $terms = Term::orderBy('start_date', 'desc')->get();
        $departments = Department::orderBy('name')->get();
        $levels = Level::orderBy('name')->get();
        $activeTerm = ActiveTermService::getActiveTerm();

        return view('admin.classes.create', compact('terms', 'departments', 'levels', 'activeTerm'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'term_id' => ['required', 'exists:terms,id'],
            'department_id' => ['required', 'exists:departments,id'],
            'level_id' => ['required', 'exists:levels,id'],
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:50', 'unique:classes,code'],
        ]);

        SchoolClass::create($request->only(['term_id', 'department_id', 'level_id', 'name', 'code']));

        return redirect()->route('admin.classes.index')
            ->with('success', 'Class created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(SchoolClass $class)
    {
        // Optimized loading - only what's needed
        $class->load([
            'term:id,name,status',
            'department:id,name',
            'level:id,name',
            'enrollments:id,student_id,class_id,status' => [
                'student:id,name,email'
            ],
            'units:id,name,code,department_id'
        ]);
        
        // Load counts separately to avoid N+1
        $stats = [
            'total_students' => $class->enrollments()->count(),
            'active_students' => $class->enrollments()->where('status', 'active')->count(),
            'total_submissions' => $class->poeSubmissions()->count(),
            'total_units' => $class->units()->count(),
        ];
        
        // Get available units from the same department that aren't already assigned
        $assignedUnitIds = $class->units->pluck('id')->toArray();
        $availableUnits = Unit::select('id', 'name', 'code')
            ->where('department_id', $class->department_id)
            ->whereNotIn('id', $assignedUnitIds)
            ->orderBy('name')
            ->get();

        return view('admin.classes.show', compact('class', 'stats', 'availableUnits'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(SchoolClass $class)
    {
        $terms = Term::orderBy('start_date', 'desc')->get();
        $departments = Department::orderBy('name')->get();
        $levels = Level::orderBy('name')->get();

        return view('admin.classes.edit', compact('class', 'terms', 'departments', 'levels'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SchoolClass $class)
    {
        $request->validate([
            'term_id' => ['required', 'exists:terms,id'],
            'department_id' => ['required', 'exists:departments,id'],
            'level_id' => ['required', 'exists:levels,id'],
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:50', 'unique:classes,code,' . $class->id],
        ]);

        $class->update($request->only(['term_id', 'department_id', 'level_id', 'name', 'code']));

        return redirect()->route('admin.classes.index')
            ->with('success', 'Class updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(SchoolClass $class)
    {
        // Check if class has enrollments or submissions
        if ($class->enrollments()->count() > 0 || $class->poeSubmissions()->count() > 0) {
            return redirect()->route('admin.classes.index')
                ->with('error', 'Cannot delete class with associated enrollments or submissions.');
        }

        $class->delete();

        return redirect()->route('admin.classes.index')
            ->with('success', 'Class deleted successfully.');
    }

    /**
     * Assign unit to class
     */
    public function assignUnit(Request $request, SchoolClass $class)
    {
        $request->validate([
            'unit_id' => ['required', 'exists:units,id'],
        ]);

        // Check if unit belongs to the same department
        $unit = Unit::findOrFail($request->unit_id);
        if ($unit->department_id != $class->department_id) {
            return redirect()->back()
                ->with('error', 'Unit must belong to the same department as the class.');
        }

        // Check if already assigned
        if ($class->units()->where('unit_id', $request->unit_id)->exists()) {
            return redirect()->back()
                ->with('error', 'This unit is already assigned to this class.');
        }

        $class->units()->attach($request->unit_id);

        return redirect()->back()
            ->with('success', 'Unit assigned to class successfully.');
    }

    /**
     * Remove unit from class
     */
    public function removeUnit(Request $request, SchoolClass $class)
    {
        $request->validate([
            'unit_id' => ['required', 'exists:units,id'],
        ]);

        // Check if there are submissions for this unit in this class
        $hasSubmissions = \App\Models\PoeSubmission::where('unit_id', $request->unit_id)
            ->where('class_id', $class->id)
            ->exists();

        if ($hasSubmissions) {
            return redirect()->back()
                ->with('error', 'Cannot remove unit from class with associated submissions.');
        }

        $class->units()->detach($request->unit_id);

        return redirect()->back()
            ->with('success', 'Unit removed from class successfully.');
    }
}
