<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Level;
use Illuminate\Http\Request;

class LevelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $levels = Level::withCount('classes')->orderBy('name')->get();
        return view('admin.levels.index', compact('levels'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.levels.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:levels'],
            'code' => ['required', 'string', 'max:50', 'unique:levels'],
            'description' => ['nullable', 'string'],
        ]);

        Level::create($request->only(['name', 'code', 'description']));

        return redirect()->route('admin.levels.index')
            ->with('success', 'Level created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Level $level)
    {
        $level->load(['classes.department', 'classes.term']);
        
        $stats = [
            'total_classes' => $level->classes()->count(),
            'active_classes' => $level->classes()->whereHas('term', function($q) {
                $q->where('status', 'active');
            })->count(),
        ];

        return view('admin.levels.show', compact('level', 'stats'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Level $level)
    {
        return view('admin.levels.edit', compact('level'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Level $level)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:levels,name,' . $level->id],
            'code' => ['required', 'string', 'max:50', 'unique:levels,code,' . $level->id],
            'description' => ['nullable', 'string'],
        ]);

        $level->update($request->only(['name', 'code', 'description']));

        return redirect()->route('admin.levels.index')
            ->with('success', 'Level updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Level $level)
    {
        // Check if level has associated classes
        if ($level->classes()->count() > 0) {
            return redirect()->route('admin.levels.index')
                ->with('error', 'Cannot delete level with associated classes.');
        }

        $level->delete();

        return redirect()->route('admin.levels.index')
            ->with('success', 'Level deleted successfully.');
    }
}
