<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Providers\RouteServiceProvider;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): View
    {
        return view('auth.login');
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $request->authenticate();

        $request->session()->regenerate();

        $user = Auth::user();

        // Check if user is active
        if (!$user->isActive()) {
            Auth::logout();
            return redirect()->route('login')
                ->withErrors(['email' => 'Your account is not active. Please contact administrator.'])
                ->withInput($request->only('email'));
        }

        // Redirect based on user's role (automatically determined from account)
        if ($user->isAdmin()) {
            return redirect()->intended(route('admin.dashboard'));
        } elseif ($user->isHOD()) {
            return redirect()->intended(route('hod.dashboard'));
        } elseif ($user->isTrainer()) {
            return redirect()->intended(route('trainer.dashboard'));
        } elseif ($user->isExamsOffice()) {
            return redirect()->intended(route('exams.dashboard'));
        } elseif ($user->isValidator()) {
            return redirect()->intended(route('validator.dashboard'));
        } elseif ($user->isStudent()) {
            return redirect()->intended(route('student.dashboard'));
        }

        return redirect()->intended(RouteServiceProvider::HOME);
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect()->route('login');
    }
}
