<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\PracticalEvidence;
use App\Models\PracticalEvidenceComment;
use App\Models\PracticalEvidenceTask;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class PracticalEvidenceController extends Controller
{
    /**
     * Display all practical evidence for students in HOD's department.
     */
    public function index(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || !$hod->department_id) {
            return view('hod.practical-evidence.index', [
                'evidences' => collect(),
                'activeTerm' => $activeTerm,
                'stats' => [
                    'total' => 0,
                    'submitted' => 0,
                    'reviewed' => 0,
                    'accepted' => 0,
                    'rejected' => 0,
                ],
                'classes' => collect(),
                'units' => collect(),
            ])->with('warning', 'No active term or department assigned.');
        }

        // Get department classes
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        if (empty($departmentClassIds)) {
            return view('hod.practical-evidence.index', [
                'evidences' => collect(),
                'activeTerm' => $activeTerm,
                'stats' => [
                    'total' => 0,
                    'submitted' => 0,
                    'reviewed' => 0,
                    'accepted' => 0,
                    'rejected' => 0,
                ],
                'classes' => collect(),
                'units' => collect(),
            ])->with('info', 'No classes found for your department.');
        }

        // Build query for practical evidence in department classes
        $query = PracticalEvidence::whereIn('class_id', $departmentClassIds)
            ->where('term_id', $activeTerm->id)
            ->with([
                'student:id,name,email,admission_number',
                'unit:id,name,code,department_id',
                'schoolClass:id,name,code',
                'comments.user:id,name'
            ]);

        // Filter by class
        if ($request->has('class') && $request->class) {
            $query->where('class_id', $request->class);
        }

        // Filter by unit
        if ($request->has('unit') && $request->unit) {
            $query->where('unit_id', $request->unit);
        }

        // Filter by status
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        // Search by student name or task name
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('task_name', 'like', '%' . $search . '%')
                  ->orWhereHas('student', function($sq) use ($search) {
                      $sq->where('name', 'like', '%' . $search . '%')
                        ->orWhere('admission_number', 'like', '%' . $search . '%');
                  });
            });
        }

        $evidences = $query->latest('uploaded_at')->paginate(20);

        // Get statistics
        $stats = [
            'total' => PracticalEvidence::whereIn('class_id', $departmentClassIds)
                ->where('term_id', $activeTerm->id)
                ->count(),
            'submitted' => PracticalEvidence::whereIn('class_id', $departmentClassIds)
                ->where('term_id', $activeTerm->id)
                ->where('status', 'submitted')
                ->count(),
            'reviewed' => PracticalEvidence::whereIn('class_id', $departmentClassIds)
                ->where('term_id', $activeTerm->id)
                ->where('status', 'reviewed')
                ->count(),
            'accepted' => PracticalEvidence::whereIn('class_id', $departmentClassIds)
                ->where('term_id', $activeTerm->id)
                ->where('status', 'accepted')
                ->count(),
            'rejected' => PracticalEvidence::whereIn('class_id', $departmentClassIds)
                ->where('term_id', $activeTerm->id)
                ->where('status', 'rejected')
                ->count(),
        ];

        // Get filter options
        $classes = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->orderBy('name')
            ->get(['id', 'name', 'code']);

        $units = Unit::where('department_id', $hod->department_id)
            ->orderBy('name')
            ->get(['id', 'name', 'code']);

        return view('hod.practical-evidence.index', compact(
            'evidences',
            'activeTerm',
            'stats',
            'classes',
            'units'
        ));
    }

    /**
     * Display a specific practical evidence.
     */
    public function show(PracticalEvidence $practicalEvidence)
    {
        $hod = Auth::user();

        // Verify HOD can access this evidence (must be in their department)
        if (!$hod->department_id || $practicalEvidence->schoolClass->department_id !== $hod->department_id) {
            abort(403, 'You are not authorized to view this evidence.');
        }

        $practicalEvidence->load([
            'student:id,name,email,admission_number',
            'unit:id,name,code',
            'schoolClass:id,name,code',
            'term:id,name',
            'comments.user:id,name,email'
        ]);

        return view('hod.practical-evidence.show', compact('practicalEvidence'));
    }

    /**
     * Add a comment to practical evidence.
     */
    public function addComment(Request $request, PracticalEvidence $practicalEvidence)
    {
        $hod = Auth::user();

        // Verify HOD can comment on this evidence
        if (!$hod->department_id || $practicalEvidence->schoolClass->department_id !== $hod->department_id) {
            abort(403, 'You are not authorized to comment on this evidence.');
        }

        $request->validate([
            'comment' => ['required', 'string', 'max:1000'],
        ]);

        // Find or create the task for this evidence
        $task = PracticalEvidenceTask::firstOrCreate(
            [
                'student_id' => $practicalEvidence->student_id,
                'unit_id' => $practicalEvidence->unit_id,
                'class_id' => $practicalEvidence->class_id,
                'task_name' => $practicalEvidence->task_name,
                'term_id' => $practicalEvidence->term_id,
            ],
            [
                'status' => 'submitted',
                'submitted_at' => $practicalEvidence->uploaded_at ?? now(),
            ]
        );

        PracticalEvidenceComment::create([
            'practical_evidence_id' => $practicalEvidence->id,
            'task_id' => $task->id,
            'user_id' => $hod->id,
            'role' => 'hod',
            'comment' => $request->comment,
        ]);

        return redirect()->back()
            ->with('success', 'Comment added successfully.');
    }

    /**
     * Download practical evidence file.
     */
    public function download(PracticalEvidence $practicalEvidence)
    {
        $hod = Auth::user();

        // Verify HOD can download this evidence
        if (!$hod->department_id || $practicalEvidence->schoolClass->department_id !== $hod->department_id) {
            abort(403, 'You are not authorized to download this evidence.');
        }

        if (!Storage::disk('public')->exists($practicalEvidence->file_path)) {
            return redirect()->back()
                ->with('error', 'File not found.');
        }

        return Storage::disk('public')->download($practicalEvidence->file_path, $practicalEvidence->original_name);
    }
}
