<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\Unit;
use App\Models\SchoolClass;
use App\Models\Assignment;
use App\Models\PoeSubmission;
use App\Models\User;
use App\Models\Level;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class UnitController extends Controller
{
    /**
     * Display a listing of units allocated to the HOD.
     */
    public function index()
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return view('hod.units.index', [
                'units' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found.');
        }

        if (!$hod->department_id) {
            return view('hod.units.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('warning', 'You are not assigned to a department.');
        }

        // Get department classes for active term
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        // Get only units allocated to this HOD (via trainer_unit_class)
        $allocatedUnitIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $hod->id)
            ->whereIn('class_id', $departmentClassIds)
            ->distinct()
            ->pluck('unit_id')
            ->toArray();

        // Get units allocated to HOD
        $units = Unit::whereIn('id', $allocatedUnitIds)
            ->where('department_id', $hod->department_id)
            ->select('id', 'name', 'code', 'description', 'department_id')
            ->with('department:id,name')
            ->orderBy('name')
            ->get();

        // Get department classes for active term
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        // Batch load statistics for all units
        $unitIds = $units->pluck('id')->toArray();
        
        // Get class assignments per unit
        $unitClassCounts = DB::table('class_unit')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->selectRaw('unit_id, count(distinct class_id) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');

        // Get assignments counts - filtered by active term
        // First, get trainer IDs for units in active term classes
        $trainerIdsByUnit = DB::table('trainer_unit_class')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->select('unit_id', 'trainer_id')
            ->distinct()
            ->get()
            ->groupBy('unit_id')
            ->map(function($items) {
                return $items->pluck('trainer_id')->unique()->toArray();
            });

        // Count assignments for each unit, filtered by active term trainers and date
        $assignmentsCounts = [];
        foreach ($unitIds as $unitId) {
            $trainerIds = $trainerIdsByUnit[$unitId] ?? [];
            if (empty($trainerIds)) {
                $assignmentsCounts[$unitId] = 0;
            } else {
                $assignmentsCounts[$unitId] = Assignment::where('unit_id', $unitId)
                    ->whereIn('trainer_id', $trainerIds)
                    ->where('created_at', '>=', $activeTerm->start_date)
                    ->where('created_at', '<=', $activeTerm->end_date ?? now())
                    ->count();
            }
        }

        // Get submissions counts
        $submissionsCounts = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');

        // Get trainer counts per unit
        $trainerCounts = DB::table('trainer_unit_class')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->selectRaw('unit_id, count(distinct trainer_id) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');

        // Attach statistics to units
        $units = $units->map(function($unit) use ($unitClassCounts, $assignmentsCounts, $submissionsCounts, $trainerCounts) {
            $unit->classes_count = $unitClassCounts[$unit->id] ?? 0;
            $unit->assignments_count = $assignmentsCounts[$unit->id] ?? 0;
            $unit->submissions_count = $submissionsCounts[$unit->id] ?? 0;
            $unit->trainers_count = $trainerCounts[$unit->id] ?? 0;
            return $unit;
        });

        return view('hod.units.index', compact('units', 'activeTerm'));
    }

    /**
     * Display the specified unit.
     */
    public function show(Unit $unit)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        // Verify unit belongs to HOD's department
        if ($unit->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to view this unit.');
        }

        // Load unit relationships
        $unit->load(['department:id,name,code', 'configuredBy:id,name']);

        // If no active term, show empty state
        if (!$activeTerm) {
            return view('hod.units.show', [
                'unit' => $unit,
                'assignedClasses' => collect(),
                'assignments' => new \Illuminate\Pagination\LengthAwarePaginator(
                    collect(),
                    0,
                    10,
                    1
                ),
                'submissions' => new \Illuminate\Pagination\LengthAwarePaginator(
                    collect(),
                    0,
                    10,
                    1
                ),
                'trainers' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found. Only showing unit configuration details.');
        }

        // Get department classes for active term
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        // Get assigned classes
        $assignedClasses = SchoolClass::whereIn('id', $departmentClassIds)
            ->whereHas('units', function($q) use ($unit) {
                $q->where('units.id', $unit->id);
            })
            ->select('id', 'name', 'code', 'term_id', 'department_id', 'level_id')
            ->with(['term:id,name', 'level:id,name'])
            ->get();

        // Get assignments for this unit in the active term (include HOD so their created assignments show)
        $trainerIds = DB::table('trainer_unit_class')
            ->where('unit_id', $unit->id)
            ->whereIn('class_id', $departmentClassIds)
            ->distinct()
            ->pluck('trainer_id')
            ->toArray();
        if (!in_array($hod->id, $trainerIds)) {
            $trainerIds[] = $hod->id;
        }

        // Only show assignments created by trainers (and HOD) in the active term
        // Also filter by creation date to be within the active term period
        $assignments = Assignment::where('unit_id', $unit->id)
            ->whereIn('trainer_id', $trainerIds)
            ->where('created_at', '>=', $activeTerm->start_date)
            ->where('created_at', '<=', $activeTerm->end_date ?? now())
            ->with('trainer:id,name')
            ->latest()
            ->paginate(10);

        // Get submissions
        $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
            ->where('unit_id', $unit->id)
            ->whereIn('class_id', $departmentClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['student:id,name,email', 'schoolClass:id,name'])
            ->latest()
            ->paginate(10);

        // Get trainers assigned to this unit
        $trainers = DB::table('trainer_unit_class')
            ->join('users', 'trainer_unit_class.trainer_id', '=', 'users.id')
            ->where('trainer_unit_class.unit_id', $unit->id)
            ->whereIn('trainer_unit_class.class_id', $departmentClassIds)
            ->select('users.id', 'users.name', 'users.email')
            ->distinct()
            ->get();

        return view('hod.units.show', compact('unit', 'assignedClasses', 'assignments', 'submissions', 'trainers', 'activeTerm'));
    }

    /**
     * Show the form for creating a new unit.
     */
    public function create()
    {
        $hod = Auth::user();
        
        if (!$hod->department_id) {
            return redirect()->route('hod.units.index')
                ->with('error', 'You are not assigned to a department.');
        }

        // Get HOD's department
        $department = \App\Models\Department::findOrFail($hod->department_id);

        return view('hod.units.create', compact('department'));
    }

    /**
     * Store a newly created unit in storage.
     */
    public function store(Request $request)
    {
        $hod = Auth::user();
        
        if (!$hod->department_id) {
            return redirect()->route('hod.units.index')
                ->with('error', 'You are not assigned to a department.');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:50', 'unique:units,code'],
            'description' => ['nullable', 'string'],
            'classification' => ['required', 'in:core_unit,basic_unit,common_unit'],
            'theory_count' => ['required', 'integer', 'min:0', 'max:20'],
            'practical_count' => ['required', 'integer', 'min:0', 'max:20'],
            'oral_count' => ['required', 'integer', 'min:0', 'max:20'],
            'project_count' => ['required', 'integer', 'min:0', 'max:20'],
        ]);

        $assessmentStructure = [
            'theory' => (int) $request->theory_count,
            'practical' => (int) $request->practical_count,
            'oral' => (int) $request->oral_count,
            'project' => (int) $request->project_count,
        ];

        // Force department_id to be HOD's department
        Unit::create([
            'name' => $request->name,
            'code' => $request->code,
            'description' => $request->description,
            'department_id' => $hod->department_id, // Always use HOD's department
            'classification' => $request->classification,
            'assessment_structure' => $assessmentStructure,
            'configured_by' => $hod->id,
            'configured_at' => now(),
        ]);

        return redirect()->route('hod.units.departmental')
            ->with('success', 'Unit created successfully.');
    }

    /**
     * Display all units in the department (not just allocated ones).
     */
    public function departmental(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return view('hod.units.departmental', [
                'units' => collect(),
                'activeTerm' => null,
                'classes' => collect(),
                'levels' => collect(),
                'trainers' => collect(),
            ])->with('warning', 'No active term found.');
        }

        if (!$hod->department_id) {
            return view('hod.units.departmental', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
                'classes' => collect(),
                'levels' => collect(),
                'trainers' => collect(),
            ])->with('warning', 'You are not assigned to a department.');
        }

        // Get department classes for active term
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        // Start with all units in HOD's department
        $unitsQuery = Unit::where('department_id', $hod->department_id);

        // Filter by class
        if ($request->has('class') && $request->class) {
            $classId = $request->class;
            $unitsQuery->whereHas('classes', function($q) use ($classId) {
                $q->where('classes.id', $classId);
            });
        }

        // Filter by level (through classes)
        if ($request->has('level') && $request->level) {
            $levelId = $request->level;
            $classIdsForLevel = SchoolClass::where('term_id', $activeTerm->id)
                ->where('department_id', $hod->department_id)
                ->where('level_id', $levelId)
                ->pluck('id')
                ->toArray();
            
            if (!empty($classIdsForLevel)) {
                $unitsQuery->whereHas('classes', function($q) use ($classIdsForLevel) {
                    $q->whereIn('classes.id', $classIdsForLevel);
                });
            } else {
                // No classes for this level, return empty
                $unitsQuery->whereRaw('1 = 0');
            }
        }

        // Filter by trainer (units allocated to specific trainer)
        if ($request->has('trainer') && $request->trainer) {
            $trainerId = $request->trainer;
            $unitIdsForTrainer = DB::table('trainer_unit_class')
                ->where('trainer_id', $trainerId)
                ->whereIn('class_id', $departmentClassIds)
                ->distinct()
                ->pluck('unit_id')
                ->toArray();
            
            if (!empty($unitIdsForTrainer)) {
                $unitsQuery->whereIn('id', $unitIdsForTrainer);
            } else {
                // No units for this trainer, return empty
                $unitsQuery->whereRaw('1 = 0');
            }
        }

        // Filter by unit (search by name or code)
        if ($request->has('unit') && $request->unit) {
            $unitId = $request->unit;
            $unitsQuery->where('id', $unitId);
        }

        // Get filtered units
        $units = $unitsQuery->select('id', 'name', 'code', 'description', 'department_id')
            ->with('department:id,name')
            ->orderBy('name')
            ->get();

        // Batch load statistics for all units
        $unitIds = $units->pluck('id')->toArray();
        
        // Get class assignments per unit
        $unitClassCounts = DB::table('class_unit')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->selectRaw('unit_id, count(distinct class_id) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');

        // Get assignments counts - filtered by active term
        // First, get trainer IDs for units in active term classes
        $trainerIdsByUnit = DB::table('trainer_unit_class')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->select('unit_id', 'trainer_id')
            ->distinct()
            ->get()
            ->groupBy('unit_id')
            ->map(function($items) {
                return $items->pluck('trainer_id')->unique()->toArray();
            });

        // Count assignments for each unit, filtered by active term trainers and date
        $assignmentsCounts = [];
        foreach ($unitIds as $unitId) {
            $trainerIds = $trainerIdsByUnit[$unitId] ?? [];
            if (empty($trainerIds)) {
                $assignmentsCounts[$unitId] = 0;
            } else {
                $assignmentsCounts[$unitId] = Assignment::where('unit_id', $unitId)
                    ->whereIn('trainer_id', $trainerIds)
                    ->where('created_at', '>=', $activeTerm->start_date)
                    ->where('created_at', '<=', $activeTerm->end_date ?? now())
                    ->count();
            }
        }

        // Get submissions counts
        $submissionsCounts = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');

        // Get trainer counts per unit
        $trainerCounts = DB::table('trainer_unit_class')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $departmentClassIds)
            ->selectRaw('unit_id, count(distinct trainer_id) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');

        // Get HOD allocation status for each unit
        $hodAllocatedUnitIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $hod->id)
            ->whereIn('class_id', $departmentClassIds)
            ->distinct()
            ->pluck('unit_id')
            ->toArray();

        // Attach statistics to units
        $units = $units->map(function($unit) use ($unitClassCounts, $assignmentsCounts, $submissionsCounts, $trainerCounts, $hodAllocatedUnitIds) {
            $unit->classes_count = $unitClassCounts[$unit->id] ?? 0;
            $unit->assignments_count = $assignmentsCounts[$unit->id] ?? 0;
            $unit->submissions_count = $submissionsCounts[$unit->id] ?? 0;
            $unit->trainers_count = $trainerCounts[$unit->id] ?? 0;
            $unit->is_allocated_to_hod = in_array($unit->id, $hodAllocatedUnitIds);
            return $unit;
        });

        // Get filter options (department-scoped)
        $classes = SchoolClass::whereIn('id', $departmentClassIds)
            ->select('id', 'name', 'code')
            ->orderBy('name')
            ->get();
        
        $levels = \App\Models\Level::whereHas('classes', function($q) use ($departmentClassIds) {
                $q->whereIn('id', $departmentClassIds);
            })
            ->select('id', 'name', 'code')
            ->orderBy('name')
            ->get();
        
        $trainers = User::where(function($query) {
                $query->where('role', 'trainer')
                      ->orWhere('role', 'hod');
            })
            ->where('department_id', $hod->department_id)
            ->where('status', 'active')
            ->select('id', 'name', 'email')
            ->orderBy('name')
            ->get();
        
        $allUnits = Unit::where('department_id', $hod->department_id)
            ->select('id', 'name', 'code')
            ->orderBy('name')
            ->get();

        return view('hod.units.departmental', compact(
            'units', 
            'activeTerm',
            'classes',
            'levels',
            'trainers',
            'allUnits'
        ));
    }

    /**
     * Show the form for configuring a unit's assessment structure.
     */
    public function configure(Unit $unit)
    {
        $hod = Auth::user();

        // Verify unit belongs to HOD's department
        if ($unit->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to configure this unit.');
        }

        $unit->load('department:id,name,code');

        return view('hod.units.configure', compact('unit'));
    }

    /**
     * Store unit configuration (classification and assessment structure).
     */
    public function storeConfiguration(Request $request, Unit $unit)
    {
        $hod = Auth::user();

        // Verify unit belongs to HOD's department
        if ($unit->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to configure this unit.');
        }

        $request->validate([
            'classification' => ['required', 'in:core_unit,basic_unit,common_unit'],
            'theory_count' => ['required', 'integer', 'min:0', 'max:20'],
            'practical_count' => ['required', 'integer', 'min:0', 'max:20'],
            'oral_count' => ['required', 'integer', 'min:0', 'max:20'],
            'project_count' => ['required', 'integer', 'min:0', 'max:20'],
        ]);

        $assessmentStructure = [
            'theory' => (int) $request->theory_count,
            'practical' => (int) $request->practical_count,
            'oral' => (int) $request->oral_count,
            'project' => (int) $request->project_count,
        ];

        $unit->update([
            'classification' => $request->classification,
            'assessment_structure' => $assessmentStructure,
            'configured_by' => $hod->id,
            'configured_at' => now(),
        ]);

        return redirect()->route('hod.units.show', ['unit' => $unit, 'context' => 'departmental'])
            ->with('success', 'Unit configuration saved successfully.');
    }
}
