<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class AssignmentController extends Controller
{
    /**
     * Display the specified assignment.
     */
    public function show(Assignment $assignment)
    {
        $student = Auth::user();
        
        // Verify assignment is published and student has access
        if (!$assignment->is_published) {
            abort(404, 'Assignment not found.');
        }

        // Verify student is enrolled in the class this assignment belongs to
        $enrollments = $student->enrollments()
            ->with('schoolClass:id,name')
            ->get();

        $classIds = $enrollments->pluck('class_id')->toArray();
        
        // CRITICAL: Check if assignment belongs to one of the student's enrolled classes
        if (!$assignment->class_id || !in_array($assignment->class_id, $classIds)) {
            abort(403, 'You do not have access to this assignment. This assignment is for a different class.');
        }
        
        // Also verify the unit is assigned to the student's class
        $hasUnitAccess = DB::table('class_unit')
            ->where('unit_id', $assignment->unit_id)
            ->whereIn('class_id', $classIds)
            ->exists();
        
        if (!$hasUnitAccess) {
            abort(403, 'You do not have access to this assignment.');
        }

        // Get or create student submission
        $submission = $assignment->studentSubmission($student->id);
        
        if (!$submission) {
            $submission = new AssignmentSubmission([
                'assignment_id' => $assignment->id,
                'student_id' => $student->id,
                'status' => 'draft',
            ]);
        }

        $assignment->load(['unit:id,name,code', 'trainer:id,name']);
        
        // For practical assignments, eager load assessor tool marks to check visibility
        if ($assignment->type === 'practical' && $submission->exists) {
            $submission->load('assessorToolMarks');
        }

        return view('student.assignments.show', compact('assignment', 'submission'));
    }

    /**
     * Store or update assignment submission.
     */
    public function submit(Request $request, Assignment $assignment)
    {
        $student = Auth::user();
        
        // Verify assignment is published and accessible
        if (!$assignment->is_published) {
            return redirect()->back()->with('error', 'Assignment not found.');
        }

        // Hard block submissions after due date/time
        // No new submissions or updates (draft or submitted) should be allowed when overdue
        if ($assignment->isOverdue()) {
            return redirect()->back()->with('error', 'The due date for this assignment has passed. Submissions are closed.');
        }

        // Verify assignment is available (start date has passed)
        if (!$assignment->isAvailableForStudents()) {
            return redirect()->back()->with('error', 'This assignment is not yet available. It will be available on ' . $assignment->start_date->format('M d, Y \a\t H:i') . '.');
        }

        // Verify student has access
        $enrollments = $student->enrollments()->get();
        $classIds = $enrollments->pluck('class_id')->toArray();
        
        // CRITICAL: Check if assignment belongs to one of the student's enrolled classes
        if (!$assignment->class_id || !in_array($assignment->class_id, $classIds)) {
            return redirect()->back()->with('error', 'You do not have access to this assignment. This assignment is for a different class.');
        }
        
        $hasAccess = DB::table('class_unit')
            ->where('unit_id', $assignment->unit_id)
            ->whereIn('class_id', $classIds)
            ->exists();
        
        if (!$hasAccess) {
            return redirect()->back()->with('error', 'You do not have access to this assignment.');
        }

        $validated = $request->validate([
            'feedback' => 'nullable|string',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240', // 10MB max per file
            'status' => 'required|in:draft,submitted',
        ]);

        // Handle file uploads
        $attachmentPaths = [];
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('assignment-submissions/' . $assignment->id, 'public');
                $attachmentPaths[] = $path;
            }
        }

        // Get or create submission
        $submission = AssignmentSubmission::firstOrNew([
            'assignment_id' => $assignment->id,
            'student_id' => $student->id,
        ]);

        // Update existing attachments if any
        if ($submission->attachments) {
            $attachmentPaths = array_merge($submission->attachments, $attachmentPaths);
        }

        $submission->feedback = $validated['feedback'] ?? null;
        $submission->attachments = !empty($attachmentPaths) ? $attachmentPaths : null;
        $submission->status = $validated['status'];
        
        if ($validated['status'] === 'submitted') {
            $submission->submitted_at = now();
        }
        
        $submission->save();

        $message = $validated['status'] === 'submitted' 
            ? 'Assignment submitted successfully!' 
            : 'Draft saved successfully!';

        return redirect()->route('student.assignments.show', $assignment)
            ->with('success', $message);
    }

    /**
     * Download assignment attachment (for regular attachments).
     */
    public function downloadAttachment(Assignment $assignment, $index)
    {
        $student = Auth::user();
        
        // Verify student has access - check assignment's class_id
        $enrollments = $student->enrollments()->get();
        $classIds = $enrollments->pluck('class_id')->toArray();
        
        // CRITICAL: Check if assignment belongs to one of the student's enrolled classes
        if (!$assignment->class_id || !in_array($assignment->class_id, $classIds)) {
            abort(403, 'You do not have access to this assignment. This assignment is for a different class.');
        }
        
        $hasAccess = DB::table('class_unit')
            ->where('unit_id', $assignment->unit_id)
            ->whereIn('class_id', $classIds)
            ->exists();
        
        if (!$hasAccess) {
            abort(403, 'You do not have access to this assignment.');
        }

        // Regular attachments only
        if (!$assignment->attachments || !isset($assignment->attachments[$index])) {
            abort(404, 'File not found.');
        }

        $filePath = $assignment->attachments[$index];
        
        if (!Storage::disk('public')->exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::disk('public')->download($filePath);
    }

    /**
     * Download practical tool (candidate or assessor).
     */
    public function downloadTool(Assignment $assignment, $toolType, $index)
    {
        $student = Auth::user();
        
        // Verify student has access - check assignment's class_id
        $enrollments = $student->enrollments()->get();
        $classIds = $enrollments->pluck('class_id')->toArray();
        
        // CRITICAL: Check if assignment belongs to one of the student's enrolled classes
        if (!$assignment->class_id || !in_array($assignment->class_id, $classIds)) {
            abort(403, 'You do not have access to this assignment. This assignment is for a different class.');
        }
        
        $hasAccess = DB::table('class_unit')
            ->where('unit_id', $assignment->unit_id)
            ->whereIn('class_id', $classIds)
            ->exists();
        
        if (!$hasAccess) {
            abort(403, 'You do not have access to this assignment.');
        }

        // Only practical assignments have tools
        if ($assignment->type !== 'practical') {
            abort(404, 'This assignment does not have tools.');
        }

        $fileArray = null;
        
        if ($toolType === 'candidate') {
            // Candidate tool: visible from start date
            if (!$assignment->canStudentViewCandidateTool()) {
                abort(403, 'Candidate tool is not yet available. It will be accessible from the start date.');
            }
            if (!$assignment->candidate_tool || count($assignment->candidate_tool) === 0) {
                abort(404, 'Candidate tool not found.');
            }
            $fileArray = $assignment->candidate_tool;
        } elseif ($toolType === 'assessor') {
            // Assessor tool: only visible after marks are awarded
            // CRITICAL: Students must NEVER access assessor tool before marking
            if (!$assignment->canStudentViewAssessorTool($student->id)) {
                abort(403, 'Assessor tool is not available until your submission has been graded and marks are awarded.');
            }
            
            // Prefer the completed assessor tool generated for this specific student
            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();

            if ($submission && $submission->completed_assessor_tool_path) {
                $filePath = $submission->completed_assessor_tool_path;
                
                if (!Storage::disk('public')->exists($filePath)) {
                    abort(404, 'Completed assessor tool not found.');
                }
                
                // Use assignment title for download filename
                $sanitizedTitle = $this->sanitizeFilename($assignment->title);
                $downloadName = 'assessor_tool_' . $sanitizedTitle . '.pdf';
                return Storage::disk('public')->download($filePath, $downloadName);
            }

            // Fallback: use the original assessor tool files attached to the assignment
            if (!$assignment->assessor_tool || count($assignment->assessor_tool) === 0) {
                abort(404, 'Assessor tool not found.');
            }
            $fileArray = $assignment->assessor_tool;
        } else {
            abort(404, 'Invalid tool type. Must be "candidate" or "assessor".');
        }

        if (!$fileArray || !isset($fileArray[$index])) {
            abort(404, 'File not found.');
        }

        $filePath = $fileArray[$index];
        
        if (!Storage::disk('public')->exists($filePath)) {
            abort(404, 'File not found.');
        }

        // Use assignment title for download filename
        $sanitizedTitle = $this->sanitizeFilename($assignment->title);
        $toolTypeLabel = $toolType === 'candidate' ? 'candidate_tool' : 'assessor_tool';
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $downloadName = $toolTypeLabel . '_' . $sanitizedTitle . '.' . $extension;
        
        return Storage::disk('public')->download($filePath, $downloadName);
    }

    /**
     * View student's own practical marks
     */
    public function viewPracticalMarks(Assignment $assignment)
    {
        $student = Auth::user();

        if ($assignment->type !== 'practical') {
            abort(404, 'This is not a practical assignment.');
        }

        // Check if results are released
        if (!$assignment->results_released) {
            return redirect()->route('student.assignments.show', $assignment)
                ->with('info', 'Practical marks have not been released yet.');
        }

        // Get student's submission
        $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
            ->where('student_id', $student->id)
            ->first();

        $marks = 0;
        $markingStatus = 'not_marked';
        $assessorToolMarks = collect();

        if ($submission) {
            $markingStatus = $submission->marking_status ?? 'not_marked';
            if ($submission->marks) {
                $marks = $submission->marks;
            } elseif ($submission->assessorToolMarks()->count() > 0) {
                $assessorToolMarks = $submission->assessorToolMarks()->orderBy('display_order')->get();
                $marks = $assessorToolMarks->sum('marks_obtained');
            }
        }

        return view('student.assignments.practical-marks', compact(
            'assignment',
            'submission',
            'marks',
            'markingStatus',
            'assessorToolMarks'
        ));
    }

    /**
     * Sanitize a string for use in filenames
     * 
     * @param string $title
     * @return string
     */
    private function sanitizeFilename($title)
    {
        // Convert to lowercase
        $sanitized = strtolower($title);
        
        // Replace spaces and special characters with underscores
        $sanitized = preg_replace('/[^a-z0-9]+/', '_', $sanitized);
        
        // Remove leading/trailing underscores
        $sanitized = trim($sanitized, '_');
        
        // Limit length to 50 characters
        $sanitized = substr($sanitized, 0, 50);
        
        // Remove trailing underscores after truncation
        $sanitized = rtrim($sanitized, '_');
        
        // If empty after sanitization, use a default
        if (empty($sanitized)) {
            $sanitized = 'assignment';
        }
        
        return $sanitized;
    }
}
