<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\Enrollment;
use App\Models\PoeSubmission;
use App\Services\ActiveTermService;
use Illuminate\Support\Facades\Auth;

class ClassController extends Controller
{
    /**
     * Display a listing of the student's enrolled classes.
     */
    public function index()
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return view('student.classes.index', [
                'enrollments' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found. Please contact your administrator.');
        }

        // Get student's enrollments for the active term
        $enrollments = $user->enrollments()
            ->whereHas('schoolClass', function($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with([
                'schoolClass:id,name,code,department_id,level_id,term_id' => [
                    'department:id,name,code',
                    'level:id,name',
                    'term:id,name,status',
                    'units:id,name,code,description,department_id' => [
                        'department:id,name'
                    ]
                ]
            ])
            ->orderBy('enrolled_at', 'desc')
            ->get();

        // Get submission status for each unit
        $submissions = $user->poeSubmissions()
            ->whereHas('schoolClass', function($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with(['unit:id,name,code', 'schoolClass:id,name'])
            ->get()
            ->keyBy(function($submission) {
                return $submission->unit_id . '_' . $submission->class_id;
            });

        // Enhance enrollments with submission status for each unit
        $enrollments->each(function($enrollment) use ($submissions) {
            $enrollment->schoolClass->units->each(function($unit) use ($enrollment, $submissions) {
                $key = $unit->id . '_' . $enrollment->class_id;
                $unit->submission = $submissions->get($key);
            });
        });

        // Calculate statistics
        $totalUnits = $enrollments->sum(function($enrollment) {
            return $enrollment->schoolClass->units->count();
        });

        $submittedUnits = $submissions->count();
        $pendingUnits = $totalUnits - $submittedUnits;

        $stats = [
            'total_classes' => $enrollments->count(),
            'total_units' => $totalUnits,
            'submitted_units' => $submittedUnits,
            'pending_units' => $pendingUnits,
        ];

        return view('student.classes.index', compact(
            'enrollments',
            'activeTerm',
            'stats'
        ));
    }
}
