<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\Unit;
use App\Models\Assignment;
use App\Models\PoeSubmission;
use App\Services\ActiveTermService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class UnitController extends Controller
{
    /**
     * Display a listing of the student's units.
     */
    public function index()
    {
        $student = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return view('student.units.index', [
                'units' => collect(),
                'activeTerm' => null,
            ])->with('error', 'No active term found. Please contact your administrator to activate a term.');
        }

        // Get ALL student's enrollments first (for debugging)
        $allEnrollments = $student->enrollments()
            ->with('schoolClass:id,name,code,term_id')
            ->get();

        // Get student's enrolled classes for active term
        $enrollments = $student->enrollments()
            ->whereHas('schoolClass', function($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with('schoolClass:id,name,code,term_id')
            ->get();

        $classIds = $enrollments->pluck('class_id')->toArray();

        // If no enrollments found for active term, provide detailed feedback
        if (empty($classIds)) {
            if ($allEnrollments->isNotEmpty()) {
                // Student has enrollments but not in active term
                $classDetails = $allEnrollments->filter(function($enrollment) {
                    return $enrollment->schoolClass !== null;
                })->map(function($enrollment) use ($activeTerm) {
                    $class = $enrollment->schoolClass;
                    return [
                        'name' => $class->name ?? 'Unknown',
                        'code' => $class->code ?? 'N/A',
                        'term_id' => $class->term_id ?? 'N/A',
                        'matches_active' => isset($class->term_id) && $class->term_id == $activeTerm->id
                    ];
                })->toArray();
                
                $classNames = collect($classDetails)->pluck('name')->filter()->join(', ') ?: 'Unknown classes';
                $warningMessage = 'You are enrolled in classes that do not belong to the active term (' . $activeTerm->name . ' - ID: ' . $activeTerm->id . '). ';
                $warningMessage .= 'Your classes: ' . $classNames . '. ';
                $warningMessage .= 'Please contact your administrator to update your class enrollment to the active term.';
                
                return view('student.units.index', [
                    'units' => collect(),
                    'activeTerm' => $activeTerm,
                ])->with('error', $warningMessage);
            }
            
            // No enrollments at all
            return view('student.units.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('error', 'You are not enrolled in any classes. Please contact your administrator.');
        }

        $unitIds = DB::table('class_unit')
            ->whereIn('class_id', $classIds)
            ->distinct()
            ->pluck('unit_id')
            ->toArray();

        if (empty($unitIds)) {
            // Get class names for better error message
            $classNames = $enrollments->pluck('schoolClass.name')->join(', ');
            return view('student.units.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('error', 'No units found. Your enrolled classes (' . $classNames . ') do not have any units assigned. Please contact your administrator to assign units to your classes.');
        }

        // Batch load all data upfront to avoid N+1 queries
        // Get unit-class mappings
        $unitClassMap = DB::table('class_unit')
            ->whereIn('class_id', $classIds)
            ->whereIn('unit_id', $unitIds)
            ->get()
            ->groupBy('unit_id')
            ->map(function($items) {
                return $items->pluck('class_id')->toArray();
            });
        
        // Batch load assignments counts - ONLY for student's enrolled classes
        $assignmentsCounts = Assignment::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
            ->where('is_published', true)
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');
        
        // Batch load all classes
        $allClassIds = $unitClassMap->flatten()->unique()->toArray();
        $classes = \App\Models\SchoolClass::whereIn('id', $allClassIds)
            ->where('term_id', $activeTerm->id)
            ->select('id', 'name', 'code')
            ->get()
            ->keyBy('id');
        
        // Batch check submissions for all units
        $submissionUnitIds = PoeSubmission::withoutGlobalScope('activeTerm')
            ->where('student_id', $student->id)
            ->whereIn('unit_id', $unitIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->pluck('unit_id')
            ->unique()
            ->toArray();
        
        // Get units with their details
        $units = Unit::whereIn('id', $unitIds)
            ->with(['department:id,name,code'])
            ->get()
            ->map(function($unit) use ($unitClassMap, $classes, $assignmentsCounts, $submissionUnitIds) {
                $unitClassIds = $unitClassMap[$unit->id] ?? [];
                $unit->assigned_classes = $classes->whereIn('id', $unitClassIds)->values();
                $unit->assignments_count = $assignmentsCounts[$unit->id] ?? 0;
                $unit->has_submission = in_array($unit->id, $submissionUnitIds);
                return $unit;
            })
            ->sortBy('name')
            ->values();

        return view('student.units.index', compact('units', 'activeTerm'));
    }

    /**
     * Display the specified unit with assignments.
     */
    public function show(Unit $unit)
    {
        $student = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            abort(403, 'No active term found. Please contact your administrator.');
        }
        
        // Verify student is enrolled in a class that has this unit
        $enrollments = $student->enrollments()
            ->whereHas('schoolClass', function($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with('schoolClass:id,name,code')
            ->get();

        $classIds = $enrollments->pluck('class_id');
        
        $isEnrolled = DB::table('class_unit')
            ->where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds)
            ->exists();
        
        if (!$isEnrolled) {
            abort(403, 'You are not enrolled in a class with this unit.');
        }

        // Get published assignments for this unit - ONLY for the student's enrolled classes
        // CRITICAL: Students should only see assignments from their own class, not other classes with the same unit
        $assignments = Assignment::where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
            ->where('is_published', true)
            ->with(['trainer:id,name', 'schoolClass:id,name,code', 'submissions' => function($query) use ($student) {
                $query->where('student_id', $student->id);
            }])
            ->latest()
            ->get()
            ->map(function($assignment) use ($student) {
                $assignment->student_submission = $assignment->studentSubmission($student->id);
                // Check if assignment is available for submission
                $assignment->is_available = $assignment->isAvailableForStudents();
                return $assignment;
            });

        $unit->load('department:id,name,code');

        return view('student.units.show', compact('unit', 'assignments', 'activeTerm'));
    }
}
