<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\Unit;
use App\Notifications\GeneralNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class AssignmentController extends Controller
{
    /**
     * Display the specified assignment.
     */
    public function show(Assignment $assignment)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to view this assignment.');
        }

        $assignment->load(['unit:id,name,code', 'trainer:id,name']);
        
        // Get submission statistics
        $submissionsCount = $assignment->submissions()->count();
        $submittedCount = $assignment->submissions()->where('status', 'submitted')->count();
        $gradedCount = $assignment->submissions()->whereNotNull('marks')->count();
        
        // Get submissions with student info
        $submissions = $assignment->submissions()
            ->with(['student:id,name,email,admission_number'])
            ->orderBy('submitted_at', 'desc')
            ->get();
        
        return view('trainer.assignments.show', compact('assignment', 'submissionsCount', 'submittedCount', 'gradedCount', 'submissions'));
    }

    /**
     * Show the form for editing the specified assignment.
     */
    public function edit(Assignment $assignment)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to edit this assignment.');
        }

        $assignment->load('unit:id,name,code');
        
        return view('trainer.assignments.edit', compact('assignment'));
    }

    /**
     * Store a newly created assignment.
     */
    public function store(Request $request, Unit $unit)
    {
        // Verify trainer is assigned to this unit
        $trainer = Auth::user();
        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return back()->with('error', 'No active term found.');
        }

        // Get classes where trainer is assigned to this unit for active term
        $activeTermClassIds = \App\Models\SchoolClass::where('term_id', $activeTerm->id)
            ->pluck('id')
            ->toArray();

        $assignedClassIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unit->id)
            ->whereIn('class_id', $activeTermClassIds)
            ->pluck('class_id')
            ->toArray();
        
        if (empty($assignedClassIds)) {
            abort(403, 'You are not assigned to this unit in the current term.');
        }

        // Check if unit is configured
        if (!$unit->isConfigured()) {
            return back()->with('error', 'This unit has not been configured by the HOD. Please contact your HOD to configure the assessment structure before creating assessments.');
        }

        // Check assessment structure limits (optional warning, not blocking)
        $type = $request->type;
        $structure = $unit->getAssessmentStructure();
        
        // Map assignment types to assessment structure types
        $typeMap = [
            'assignment' => 'theory',
            'exam' => 'theory',
            'practical' => 'practical',
            'project' => 'project',
        ];
        
        $assessmentType = $typeMap[$type] ?? null;
        if ($assessmentType) {
            // Get the selected class_id for counting existing assignments
            $selectedClassId = $request->class_id ?? null;
            $existingCount = Assignment::where('unit_id', $unit->id)
                ->where('trainer_id', $trainer->id)
                ->where('class_id', $selectedClassId) // Count only for the selected class
                ->where('is_published', true)
                ->whereIn('type', $type === 'assignment' || $type === 'exam' ? ['assignment', 'exam'] : [$type])
                ->count();
            
            $required = $structure[$assessmentType] ?? 0;
            
            // Allow creation but warn if exceeding requirements
            if ($existingCount >= $required) {
                // Still allow, but this is informational
            }
        }

        $rules = [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:assignment,exam,practical,project',
            'class_id' => ['required', 'exists:classes,id', function($attribute, $value, $fail) use ($assignedClassIds) {
                if (!in_array($value, $assignedClassIds)) {
                    $fail('The selected class is not assigned to you for this unit.');
                }
            }],
            'total_marks' => 'nullable|integer|min:0',
            'due_date' => 'nullable|date',
            'start_date' => 'nullable|date',
            'instructions' => 'nullable|string',
            'is_published' => 'boolean',
        ];

        $messages = [];

        // For practical assignments, require both tools
        if ($request->type === 'practical') {
            $rules['candidate_tool'] = 'required|array|min:1';
            $rules['assessor_tool'] = 'required|array|min:1';
            $rules['candidate_tool.*'] = 'file|max:10240';
            $rules['assessor_tool.*'] = 'file|max:10240';
            $messages['candidate_tool.required'] = 'Candidate Tool is required for practical assignments.';
            $messages['candidate_tool.min'] = 'At least one Candidate Tool file is required.';
            $messages['assessor_tool.required'] = 'Assessor Tool is required for practical assignments.';
            $messages['assessor_tool.min'] = 'At least one Assessor Tool file is required.';
        } else {
            // For non-practical types, allow regular attachments
            $rules['attachments'] = 'nullable|array';
            $rules['attachments.*'] = 'file|max:10240'; // 10MB max per file
            // Don't require candidate_tool or assessor_tool for non-practical types
            $rules['candidate_tool'] = 'nullable|array';
            $rules['assessor_tool'] = 'nullable|array';
            $rules['candidate_tool.*'] = 'nullable|file|max:10240';
            $rules['assessor_tool.*'] = 'nullable|file|max:10240';
        }

        $validated = $request->validate($rules, $messages);

        // Handle file uploads based on assignment type
        $attachmentPaths = [];
        $candidateToolPaths = [];
        $assessorToolPaths = [];

        if ($validated['type'] === 'practical') {
            // For practical: handle candidate and assessor tools
            // Sanitize assignment title for filename
            $sanitizedTitle = $this->sanitizeFilename($validated['title']);
            
            if ($request->hasFile('candidate_tool')) {
                $index = 0;
                foreach ($request->file('candidate_tool') as $file) {
                    $extension = $file->getClientOriginalExtension();
                    $filename = 'candidate_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $extension;
                    $path = $file->storeAs('assignments/' . $unit->id . '/candidate-tool', $filename, 'public');
                    $candidateToolPaths[] = $path;
                    $index++;
                }
            }
            if ($request->hasFile('assessor_tool')) {
                $index = 0;
                foreach ($request->file('assessor_tool') as $file) {
                    $extension = $file->getClientOriginalExtension();
                    $filename = 'assessor_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $extension;
                    $path = $file->storeAs('assignments/' . $unit->id . '/assessor-tool', $filename, 'public');
                    $assessorToolPaths[] = $path;
                    $index++;
                }
            }
        } else {
            // For other types: handle regular attachments (assignment, exam, project)
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('assignments/' . $unit->id, 'public');
                    $attachmentPaths[] = $path;
                }
            }
            // Clear any practical tool data for non-practical types
            $candidateToolPaths = null;
            $assessorToolPaths = null;
        }

        $assignment = Assignment::create([
            'unit_id' => $unit->id,
            'trainer_id' => $trainer->id,
            'class_id' => $validated['class_id'], // CRITICAL: Link assignment to specific class
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'type' => $validated['type'],
            'total_marks' => $validated['total_marks'] ?? null,
            'due_date' => $validated['due_date'] ?? null,
            'start_date' => $validated['start_date'] ?? null,
            'instructions' => $validated['instructions'] ?? null,
            'is_published' => $validated['is_published'] ?? false,
            'attachments' => !empty($attachmentPaths) ? $attachmentPaths : null,
            'candidate_tool' => !empty($candidateToolPaths) ? $candidateToolPaths : null,
            'assessor_tool' => !empty($assessorToolPaths) ? $assessorToolPaths : null,
        ]);

        if ($assignment->is_published) {
            $this->notifyStudentsInClass($assignment, $trainer, 'New assignment: ' . $assignment->title);
        }

        return redirect()->route('trainer.units.show', $unit)
            ->with('success', 'Assignment created successfully.');
    }

    /**
     * Notify all students enrolled in the assignment's class about the assignment.
     */
    private function notifyStudentsInClass(Assignment $assignment, $sender, string $message): void
    {
        $assignment->load('unit:id,name');
        $studentIds = DB::table('enrollments')
            ->where('class_id', $assignment->class_id)
            ->where('status', 'active')
            ->pluck('student_id');

        $students = \App\Models\User::whereIn('id', $studentIds)
            ->where('role', 'student')
            ->where('status', 'active')
            ->get();

        foreach ($students as $student) {
            $student->notify(new GeneralNotification(
                'New assignment',
                $message . ' — Unit: ' . ($assignment->unit->name ?? '') . '. Check your unit page to view and submit.',
                $sender->id
            ));
        }
    }

    /**
     * Update the specified assignment.
     */
    public function update(Request $request, Assignment $assignment)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to update this assignment.');
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:assignment,exam,practical,project',
            'total_marks' => 'nullable|integer|min:0',
            'due_date' => 'nullable|date',
            'start_date' => 'nullable|date',
            'instructions' => 'nullable|string',
            'is_published' => 'boolean',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240',
            'candidate_tool' => 'nullable|array',
            'candidate_tool.*' => 'file|max:10240',
            'assessor_tool' => 'nullable|array',
            'assessor_tool.*' => 'file|max:10240',
        ]);

        // Handle file uploads based on assignment type. On edit, new files REPLACE existing ones.
        $attachmentPaths = $assignment->attachments ?? [];
        $candidateToolPaths = $assignment->candidate_tool ?? [];
        $assessorToolPaths = $assignment->assessor_tool ?? [];

        if ($validated['type'] === 'practical') {
            $sanitizedTitle = $this->sanitizeFilename($validated['title']);

            if ($request->hasFile('candidate_tool')) {
                foreach ($candidateToolPaths as $path) {
                    Storage::disk('public')->delete($path);
                }
                $candidateToolPaths = [];
                $index = 0;
                foreach ($request->file('candidate_tool') as $file) {
                    $extension = $file->getClientOriginalExtension();
                    $filename = 'candidate_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $extension;
                    $path = $file->storeAs('assignments/' . $assignment->unit_id . '/candidate-tool', $filename, 'public');
                    $candidateToolPaths[] = $path;
                    $index++;
                }
            }
            if ($request->hasFile('assessor_tool')) {
                foreach ($assessorToolPaths as $path) {
                    Storage::disk('public')->delete($path);
                }
                $assessorToolPaths = [];
                $index = 0;
                foreach ($request->file('assessor_tool') as $file) {
                    $extension = $file->getClientOriginalExtension();
                    $filename = 'assessor_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $extension;
                    $path = $file->storeAs('assignments/' . $assignment->unit_id . '/assessor-tool', $filename, 'public');
                    $assessorToolPaths[] = $path;
                    $index++;
                }
            }
        } else {
            if ($request->hasFile('attachments')) {
                foreach ($attachmentPaths as $path) {
                    Storage::disk('public')->delete($path);
                }
                $attachmentPaths = [];
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('assignments/' . $assignment->unit_id, 'public');
                    $attachmentPaths[] = $path;
                }
            }
        }

        $wasPublished = $assignment->is_published;
        $assignment->update([
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'type' => $validated['type'],
            'total_marks' => $validated['total_marks'] ?? null,
            'due_date' => $validated['due_date'] ?? null,
            'start_date' => $validated['start_date'] ?? null,
            'instructions' => $validated['instructions'] ?? null,
            'is_published' => $validated['is_published'] ?? false,
            'attachments' => !empty($attachmentPaths) ? $attachmentPaths : null,
            'candidate_tool' => !empty($candidateToolPaths) ? $candidateToolPaths : null,
            'assessor_tool' => !empty($assessorToolPaths) ? $assessorToolPaths : null,
        ]);

        if ($assignment->is_published && !$wasPublished) {
            $this->notifyStudentsInClass($assignment, $trainer, 'New assignment: ' . $assignment->title);
        }

        return redirect()->route('trainer.units.show', $assignment->unit)
            ->with('success', 'Assignment updated successfully.');
    }

    /**
     * Remove the specified assignment.
     */
    public function destroy(Assignment $assignment)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to delete this assignment.');
        }

        // Delete attached files
        if ($assignment->attachments) {
            foreach ($assignment->attachments as $path) {
                Storage::disk('public')->delete($path);
            }
        }
        if ($assignment->candidate_tool) {
            foreach ($assignment->candidate_tool as $path) {
                Storage::disk('public')->delete($path);
            }
        }
        if ($assignment->assessor_tool) {
            foreach ($assignment->assessor_tool as $path) {
                Storage::disk('public')->delete($path);
            }
        }

        $unit = $assignment->unit;
        $assignment->delete();

        return redirect()->route('trainer.units.show', $unit)
            ->with('success', 'Assignment deleted successfully.');
    }

    /**
     * Download assignment attachment.
     */
    public function downloadAttachment(Assignment $assignment, $index)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to download this attachment.');
        }

        if (!$assignment->attachments || !isset($assignment->attachments[$index])) {
            abort(404, 'File not found.');
        }

        $filePath = $assignment->attachments[$index];
        
        if (!Storage::disk('public')->exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::disk('public')->download($filePath);
    }

    /**
     * Download practical tool (candidate or assessor).
     */
    public function downloadTool(Assignment $assignment, $toolType, $index)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to download this tool.');
        }

        // Only practical assignments have tools
        if ($assignment->type !== 'practical') {
            abort(404, 'This assignment does not have tools.');
        }

        $fileArray = null;

        if ($toolType === 'candidate') {
            $fileArray = $assignment->candidate_tool;
        } elseif ($toolType === 'assessor') {
            $fileArray = $assignment->assessor_tool;
        } else {
            abort(404, 'Invalid tool type.');
        }

        if (!$fileArray || !isset($fileArray[$index])) {
            abort(404, 'File not found.');
        }

        $filePath = $fileArray[$index];
        
        if (!Storage::disk('public')->exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::disk('public')->download($filePath);
    }

    /**
     * Show the form for marking a submission.
     */
    public function showSubmission(Assignment $assignment, AssignmentSubmission $submission)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to view this submission.');
        }
        
        // Verify submission belongs to assignment
        if ($submission->assignment_id !== $assignment->id) {
            abort(404, 'Submission not found for this assignment.');
        }
        
        // Only allow marking for non-practical assignments
        if ($assignment->type === 'practical') {
            abort(403, 'Practical assignments use a different marking interface.');
        }
        
        $submission->load(['student:id,name,email,admission_number', 'assignment:id,title,type,total_marks']);
        
        return view('trainer.assignments.mark-submission', compact('assignment', 'submission'));
    }

    /**
     * Update marks for a submission.
     */
    public function markSubmission(Request $request, Assignment $assignment, AssignmentSubmission $submission)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to mark this submission.');
        }
        
        // Verify submission belongs to assignment
        if ($submission->assignment_id !== $assignment->id) {
            abort(404, 'Submission not found for this assignment.');
        }
        
        // Only allow marking for non-practical assignments
        if ($assignment->type === 'practical') {
            abort(403, 'Practical assignments use a different marking interface.');
        }
        
        $maxMarks = $assignment->total_marks ?? 100;
        
        $validated = $request->validate([
            'marks' => ['required', 'numeric', 'min:0', 'max:' . $maxMarks],
            'trainer_feedback' => ['nullable', 'string', 'max:5000'],
        ]);
        
        $submission->update([
            'marks' => $validated['marks'],
            'trainer_feedback' => $validated['trainer_feedback'] ?? null,
            'status' => 'graded',
            // keep marking_status in sync so validation readiness and reports see this as marked
            'marking_status' => 'marked',
            'graded_at' => now(),
        ]);
        
        return redirect()->route('trainer.assignments.show', $assignment)
            ->with('success', 'Submission marked successfully.');
    }

    /**
     * Sanitize a string for use in filenames
     * 
     * @param string $title
     * @return string
     */
    private function sanitizeFilename($title)
    {
        // Convert to lowercase
        $sanitized = strtolower($title);
        
        // Replace spaces and special characters with underscores
        $sanitized = preg_replace('/[^a-z0-9]+/', '_', $sanitized);
        
        // Remove leading/trailing underscores
        $sanitized = trim($sanitized, '_');
        
        // Limit length to 50 characters
        $sanitized = substr($sanitized, 0, 50);
        
        // Remove trailing underscores after truncation
        $sanitized = rtrim($sanitized, '_');
        
        // If empty after sanitization, use a default
        if (empty($sanitized)) {
            $sanitized = 'assignment';
        }
        
        return $sanitized;
    }
}
