<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Models\ValidationRequest;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class MarksController extends Controller
{
    /**
     * Display marks management page - shows list of units.
     */
    public function index(Request $request)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return view('trainer.marks.index', [
                'units' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found.');
        }

        // Limit to classes in the active term
        $activeTermClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->pluck('id')
            ->toArray();

        // All trainer-unit-class allocations for this trainer in the active term
        $allocations = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->whereIn('class_id', $activeTermClassIds)
            ->select('unit_id', 'class_id')
            ->get();

        if ($allocations->isEmpty()) {
            return view('trainer.marks.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('info', 'You don\'t have any units assigned for the current term.');
        }

        $unitIds = $allocations->pluck('unit_id')->unique()->toArray();

        // Load base units
        $unitsQuery = Unit::whereIn('id', $unitIds)
            ->with(['department:id,name,code']);

        // Filter by search
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $unitsQuery->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%");
            });
        }

        $baseUnits = $unitsQuery->orderBy('name')->get()->keyBy('id');

        // Filter allocations to only units that passed the search filter
        $allocations = $allocations->filter(function ($row) use ($baseUnits) {
            return $baseUnits->has($row->unit_id);
        });

        if ($allocations->isEmpty()) {
            return view('trainer.marks.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('info', 'No units match your search for the current term.');
        }

        $classIds = $allocations->pluck('class_id')->unique()->toArray();

        // Preload classes
        $classes = SchoolClass::whereIn('id', $classIds)
            ->where('term_id', $activeTerm->id)
            ->select('id', 'name', 'code')
            ->get()
            ->keyBy('id');

        // Preload assignment counts per (unit, class)
        $assignmentCounts = Assignment::where('trainer_id', $trainer->id)
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds)
            ->where('is_published', true)
            ->selectRaw('unit_id, class_id, count(*) as count')
            ->groupBy('unit_id', 'class_id')
            ->get()
            ->keyBy(function ($row) {
                return $row->unit_id.'_'.$row->class_id;
            });

        // Build one view model per (unit, class) allocation
        $units = $allocations->map(function ($row) use ($baseUnits, $classes, $assignmentCounts) {
            $unit = $baseUnits->get($row->unit_id);
            if (! $unit) {
                return null;
            }

            $clone = clone $unit;
            $clone->allocation_class_id = $row->class_id;

            $class = $classes->get($row->class_id);
            $clone->assigned_class = $class;

            $key = $row->unit_id.'_'.$row->class_id;
            $clone->assignments_count = $assignmentCounts->has($key)
                ? $assignmentCounts->get($key)->count
                : 0;

            // In marks management context, classes_count is always 1 (this specific class)
            $clone->classes_count = 1;

            return $clone;
        })
        ->filter()
        ->sortBy(function ($u) {
            $className = optional($u->assigned_class)->name ?? '';
            return $u->name.' '.$className;
        })
        ->values();

        return view('trainer.marks.index', compact('units', 'activeTerm'));
    }

    /**
     * Show unit details with validation readiness and marks printing options.
     */
    public function showUnit(Request $request, Unit $unit)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return back()->with('error', 'No active term found.');
        }

        // Verify trainer is assigned to this unit
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unit->id)
            ->exists();

        if (!$isAssigned) {
            abort(403, 'You are not assigned to this unit.');
        }

        // Get classes assigned to this trainer for this unit
        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unit->id)
            ->pluck('class_id');

        $classes = \App\Models\SchoolClass::whereIn('id', $classIds)
            ->where('term_id', $activeTerm->id)
            ->orderBy('name')
            ->get();

        // Get selected class from request
        $selectedClass = null;
        $validationReadiness = null;
        
        $pendingValidationRequest = null;
        if ($request->has('class') && $request->class) {
            $selectedClass = $classes->find($request->class);
            if ($selectedClass) {
                $validationReadiness = $this->getValidationReadiness($selectedClass, $unit);
                $activeTerm = ActiveTermService::getActiveTerm();
                if ($activeTerm) {
                    $pendingValidationRequest = ValidationRequest::where('class_id', $selectedClass->id)
                        ->where('unit_id', $unit->id)
                        ->where('term_id', $activeTerm->id)
                        ->where('status', 'pending')
                        ->first();
                }
            }
        }

        // Get assignments for this unit - filtered by trainer's assigned classes
        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unit->id)
            ->pluck('class_id')
            ->toArray();
        
        $assignments = Assignment::where('unit_id', $unit->id)
            ->where('trainer_id', $trainer->id)
            ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for trainer's assigned classes
            ->where('is_published', true)
            ->with(['unit:id,name,code', 'schoolClass:id,name,code'])
            ->orderBy('created_at', 'desc')
            ->get();

        // Load unit relationships
        $unit->load(['department:id,name,code', 'configuredBy:id,name']);

        return view('trainer.marks.show-unit', compact(
            'unit',
            'classes',
            'selectedClass',
            'validationReadiness',
            'assignments',
            'activeTerm',
            'pendingValidationRequest'
        ));
    }

    /**
     * Get validation readiness status for a class + unit.
     */
    public function getValidationReadiness(SchoolClass $class, Unit $unit)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $class->term_id !== $activeTerm->id) {
            return [
                'ready' => false,
                'errors' => ['Invalid or inactive term.'],
            ];
        }

        // Ensure trainer is assigned
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('class_id', $class->id)
            ->where('unit_id', $unit->id)
            ->exists();

        if (!$isAssigned) {
            return [
                'ready' => false,
                'errors' => ['You are not assigned to this class and unit.'],
            ];
        }

        // Check if unit is configured
        if (!$unit->isConfigured()) {
            return [
                'ready' => false,
                'errors' => ['Unit assessment structure has not been configured by HOD.'],
                'unit_not_configured' => true,
            ];
        }

        // Get active students
        $studentIds = $class->students()
            ->wherePivot('status', 'active')
            ->pluck('users.id');

        if ($studentIds->isEmpty()) {
            return [
                'ready' => false,
                'errors' => ['No active students found in this class.'],
            ];
        }

        $expectedCount = $studentIds->count();
        $errors = [];
        $assessmentStatus = [];
        $structure = $unit->getAssessmentStructure();

        // Check each assessment type requirement
        foreach (['theory', 'practical', 'oral', 'project'] as $type) {
            $required = $structure[$type] ?? 0;
            
            if ($required > 0) {
                // Map type to assignment type
                $assignmentTypeMap = [
                    'theory' => ['assignment', 'exam'],
                    'practical' => ['practical'],
                    'oral' => ['practical'], // Oral is part of practical
                    'project' => ['project'],
                ];
                
                $typesToCheck = $assignmentTypeMap[$type] ?? [$type];
                
                // Count completed assessments of this type - filtered by this specific class
                $completedCount = Assignment::where('unit_id', $unit->id)
                    ->where('trainer_id', $trainer->id)
                    ->where('class_id', $class->id) // CRITICAL: Only assignments for this specific class
                    ->where('is_published', true)
                    ->whereIn('type', $typesToCheck)
                    ->get()
                    ->filter(function($assignment) use ($studentIds, $expectedCount) {
                        // Check if all students are marked for this assignment
                        $markedCount = AssignmentSubmission::where('assignment_id', $assignment->id)
                            ->whereIn('student_id', $studentIds)
                            ->whereIn('marking_status', ['marked', 'absent'])
                            ->count();
                        return $markedCount >= $expectedCount;
                    })
                    ->count();

                $assessmentStatus[$type] = [
                    'required' => $required,
                    'completed' => $completedCount,
                    'status' => $completedCount >= $required ? 'complete' : 'incomplete',
                ];

                if ($completedCount < $required) {
                    $missing = $required - $completedCount;
                    $typeLabel = ucfirst($type);
                    $errors[] = "Missing: {$missing} {$typeLabel} Assessment(s) (Required: {$required}, Completed: {$completedCount})";
                }
            } else {
                $assessmentStatus[$type] = [
                    'required' => 0,
                    'completed' => 0,
                    'status' => 'complete',
                ];
            }
        }

        // Additional check: ensure all published assignments have all students marked
        $allAssignments = Assignment::where('unit_id', $unit->id)
            ->where('trainer_id', $trainer->id)
            ->where('class_id', $class->id) // CRITICAL: Only assignments for this specific class
            ->where('is_published', true)
            ->get();

        foreach ($allAssignments as $assignment) {
            $markedCount = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->whereIn('student_id', $studentIds)
                ->whereIn('marking_status', ['marked', 'absent'])
                ->count();

            if ($markedCount < $expectedCount) {
                $missing = $expectedCount - $markedCount;
                $errors[] = "{$assignment->title} ({$assignment->type}): {$missing} student(s) missing marks.";
            }
        }

        return [
            'ready' => empty($errors),
            'errors' => $errors,
            'assessment_status' => $assessmentStatus,
            'unit_classification' => $unit->classification,
            'unit_name' => $unit->name,
        ];
    }

    /**
     * Push a class (unit + term) to validation.
     */
    public function pushClassToValidation(Request $request, SchoolClass $class, Unit $unit)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $class->term_id !== $activeTerm->id) {
            return back()->with('validation_push_error', 'Cannot push class to validation: invalid or inactive term.');
        }

        // Ensure trainer is assigned to this class + unit
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('class_id', $class->id)
            ->where('unit_id', $unit->id)
            ->exists();

        if (!$isAssigned) {
            abort(403, 'You are not assigned to this class and unit.');
        }

        // Check if already pending/approved (locked)
        if (ValidationRequest::isLocked($class->id, $unit->id, $activeTerm->id)) {
            return back()->with('validation_push_error', 'This class is already under validation or approved for this unit.');
        }

        // Get validation readiness
        $readiness = $this->getValidationReadiness($class, $unit);

        if (!$readiness['ready']) {
            return back()
                ->with('validation_push_error', 'Class cannot be pushed to validation. Please review the requirements.')
                ->with('validation_errors', $readiness['errors'])
                ->with('validation_readiness', $readiness);
        }

        // Check if there's already a validation request for this class (from another trainer/unit)
        $existingClassRequest = ValidationRequest::where('class_id', $class->id)
            ->where('term_id', $activeTerm->id)
            ->where('status', 'pending')
            ->first();

        ValidationRequest::create([
            'class_id' => $class->id,
            'unit_id' => $unit->id,
            'term_id' => $activeTerm->id,
            'submitted_by' => $trainer->id,
            'submitted_role' => 'trainer',
            'status' => 'pending',
            'submitted_at' => now(),
        ]);

        // TODO: Notify validators about new validation request

        if ($existingClassRequest) {
            return back()->with('success', 'Unit has been added to the existing class validation request.');
        } else {
            return back()->with('success', 'Class has been pushed to the validation portal successfully.');
        }
    }

    /**
     * Cancel a pending validation request (trainer can cancel for their assigned class+unit).
     */
    public function cancelValidationRequest(Request $request, ValidationRequest $validationRequest)
    {
        $trainer = Auth::user();

        if ($validationRequest->status !== 'pending') {
            return back()->with('error', 'Only pending validation requests can be cancelled.');
        }

        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('class_id', $validationRequest->class_id)
            ->where('unit_id', $validationRequest->unit_id)
            ->exists();

        if (!$isAssigned) {
            abort(403, 'You can only cancel validation requests for classes and units you are assigned to.');
        }

        $validationRequest->delete();
        return back()->with('success', 'Validation request has been cancelled. The unit is no longer in the validation portal.');
    }

    /**
     * Print unit marks for a specific assignment
     */
    public function printAssignment(Request $request, Assignment $assignment)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You are not authorized to view this assignment.');
        }

        // Get all submissions for this assignment
        $submissionsQuery = AssignmentSubmission::where('assignment_id', $assignment->id)
            ->with([
                'student:id,name,email,admission_number',
                'assessorToolMarks'
            ]);
        
        // Filter by class if provided
        if ($request->has('class') && $request->class) {
            $submissionsQuery->whereHas('student.enrollments', function($q) use ($request) {
                $q->where('class_id', $request->class)
                  ->where('status', 'active');
            });
        }
        
        $submissions = $submissionsQuery->orderBy('student_id')->get();

        // Get unit and class information
        $assignment->load(['unit:id,name,code,department_id' => ['department:id,name'], 'trainer:id,name']);
        
        // Get classes for this unit
        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $assignment->unit_id)
            ->pluck('class_id');
        
        $classes = \App\Models\SchoolClass::whereIn('id', $classIds)
            ->select('id', 'name', 'code')
            ->get();

        // Get institution settings for letterhead
        $institutionSettings = \App\Models\Setting::where('key', 'like', 'institution.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        $assessorStructure = \App\Models\AssessorToolStructure::where('assignment_id', $assignment->id)->first();

        // Detect practical/oral tasks from the parsed assessor tool structure.
        // We look at row_identifiers like "task_1_practical_item_3" / "task_1_oral_item_2"
        // and aggregate available marks per task for each type.
        $practicalMaxPerIndex = [];
        $oralMaxPerIndex = [];
        $practicalColumns = [];
        $oralColumns = [];
        $practicalTaskIndexMap = [];
        $oralTaskIndexMap = [];

        if ($assessorStructure && is_array($assessorStructure->structure ?? null)) {
            $structureData = $assessorStructure->structure;
            $sections = $structureData['sections'] ?? [];

            // Raw totals keyed by original task number from the document
            $practicalTaskMax = [];
            $oralTaskMax = [];

            foreach ($sections as $row) {
                $rowId = $row['row_identifier'] ?? '';
                $marksAvailable = (float) ($row['marks_available'] ?? 0);

                if (!$rowId || $marksAvailable <= 0) {
                    continue;
                }

                $isPractical = preg_match('/practical/i', $rowId);
                $isOral = preg_match('/oral/i', $rowId);

                if (!$isPractical && !$isOral) {
                    continue;
                }

                if (preg_match('/task_(\d+)/i', $rowId, $m)) {
                    $taskIdx = max(1, (int) $m[1]);

                    if ($isPractical) {
                        $practicalTaskMax[$taskIdx] = ($practicalTaskMax[$taskIdx] ?? 0) + $marksAvailable;
                    } elseif ($isOral) {
                        $oralTaskMax[$taskIdx] = ($oralTaskMax[$taskIdx] ?? 0) + $marksAvailable;
                    }
                }
            }

            // Renormalise task numbers so the first detected task becomes index 0 (P1/O1),
            // regardless of the original numeric suffix in the Word document.
            if (!empty($practicalTaskMax)) {
                $practicalTaskIds = array_keys($practicalTaskMax);
                sort($practicalTaskIds);
                foreach ($practicalTaskIds as $seqIndex => $taskId) {
                    $practicalTaskIndexMap[$taskId] = $seqIndex; // 0-based
                }

                $practicalTaskCount = count($practicalTaskIds);
                $practicalMaxPerIndex = array_fill(0, $practicalTaskCount, 0);
                foreach ($practicalTaskMax as $taskId => $maxMarks) {
                    $index = $practicalTaskIndexMap[$taskId];
                    $practicalMaxPerIndex[$index] += $maxMarks;
                }
                // Column labels P1, P2, ...
                $practicalColumns = range(1, $practicalTaskCount);
            }

            if (!empty($oralTaskMax)) {
                $oralTaskIds = array_keys($oralTaskMax);
                sort($oralTaskIds);
                foreach ($oralTaskIds as $seqIndex => $taskId) {
                    $oralTaskIndexMap[$taskId] = $seqIndex; // 0-based
                }

                $oralTaskCount = count($oralTaskIds);
                $oralMaxPerIndex = array_fill(0, $oralTaskCount, 0);
                foreach ($oralTaskMax as $taskId => $maxMarks) {
                    $index = $oralTaskIndexMap[$taskId];
                    $oralMaxPerIndex[$index] += $maxMarks;
                }
                // Column labels O1, O2, ...
                $oralColumns = range(1, $oralTaskCount);
            }
        }

        // Fallback if structure is missing or doesn't contain task information:
        // treat the whole practical as a single practical task (no oral split).
        if (empty($practicalColumns) && empty($oralColumns)) {
            if ($assignment->type === 'oral') {
                $oralColumns = [1];
                $oralMaxPerIndex = [($assignment->total_marks ?: 100)];
                $oralTaskIndexMap = [1 => 0];
            } else {
                $practicalColumns = [1];
                $practicalMaxPerIndex = [($assignment->total_marks ?: 100)];
                $practicalTaskIndexMap = [1 => 0];
            }
        }

        $practicalTaskCount = count($practicalColumns);
        $oralTaskCount = count($oralColumns);

        $practicalTotalMax = array_sum($practicalMaxPerIndex);
        $oralTotalMax = array_sum($oralMaxPerIndex);

        $taskMarks = [];
        if ($assessorStructure) {
            $marks = \App\Models\AssessorToolMark::whereIn('assignment_submission_id', $submissions->pluck('id'))
                ->get()
                ->groupBy('assignment_submission_id');

            foreach ($submissions as $submission) {
                $taskMarks[$submission->id] = [
                    'P' => array_fill(0, $practicalTaskCount, 0),
                    'O' => array_fill(0, $oralTaskCount, 0),
                ];

                $submissionMarks = $marks->get($submission->id);
                if (!$submissionMarks) {
                    continue;
                }

                foreach ($submissionMarks as $mark) {
                    $row = $mark->row_identifier;
                    $type = null;
                    $taskIdx = null;

                    // New-style structures: row identifiers contain PRACTICAL / ORAL and TASK_#
                    if (preg_match('/practical/i', $row)) {
                        $type = 'P';
                        if (preg_match('/task_(\d+)/i', $row, $m)) {
                            $taskIdx = max(1, (int) $m[1]);
                        }
                    } elseif (preg_match('/oral/i', $row)) {
                        $type = 'O';
                        if (preg_match('/task_(\d+)/i', $row, $m)) {
                            $taskIdx = max(1, (int) $m[1]);
                        }
                    }

                    if ($type && $taskIdx) {
                        // Map by task index when the structure encodes tasks explicitly
                        if ($type === 'P' && isset($practicalTaskIndexMap[$taskIdx])) {
                            $index = $practicalTaskIndexMap[$taskIdx];
                            if ($index < $practicalTaskCount) {
                                $taskMarks[$submission->id]['P'][$index] += $mark->marks_obtained;
                            }
                        } elseif ($type === 'O' && isset($oralTaskIndexMap[$taskIdx])) {
                            $index = $oralTaskIndexMap[$taskIdx];
                            if ($index < $oralTaskCount) {
                                $taskMarks[$submission->id]['O'][$index] += $mark->marks_obtained;
                            }
                        }
                    } else {
                        // Fallback for legacy assessor tool structures that don't encode
                        // task numbers or PRACTICAL/ORAL in the row_identifier.
                        // In that case we place all marks into the first practical or oral column.
                        if ($practicalTaskCount > 0 && $assignment->type === 'practical') {
                            $taskMarks[$submission->id]['P'][0] += $mark->marks_obtained;
                        } elseif ($oralTaskCount > 0 && $assignment->type === 'oral') {
                            $taskMarks[$submission->id]['O'][0] += $mark->marks_obtained;
                        }
                    }
                }
            }
        }

        return view('trainer.marks.print-assignment', compact(
            'assignment',
            'submissions',
            'classes',
            'activeTerm',
            'institutionSettings',
            'trainer',
            'practicalColumns',
            'oralColumns',
            'taskMarks',
            'practicalMaxPerIndex',
            'oralMaxPerIndex',
            'practicalTotalMax',
            'oralTotalMax'
        ));
    }

    /**
     * Print unit marks summary
     */
    public function printUnitMarks(Request $request, $unitId)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        // Verify trainer is assigned to this unit
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unitId)
            ->exists();
        
        if (!$isAssigned) {
            abort(403, 'You are not assigned to this unit.');
        }

        $unit = \App\Models\Unit::with('department:id,name')->findOrFail($unitId);

        // Determine which class(es) to include. If a specific class_id is provided,
        // restrict the report to that class only so different classes for the same
        // unit are not combined.
        $classIdsQuery = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unitId);

        $selectedClass = null;
        if ($request->filled('class_id')) {
            $requestedClassId = (int) $request->query('class_id');
            $classIdsQuery->where('class_id', $requestedClassId);
            $selectedClass = \App\Models\SchoolClass::select('id', 'name', 'code')
                ->find($requestedClassId);
        }

        $classIds = $classIdsQuery->pluck('class_id');

        if ($classIds->isEmpty()) {
            abort(403, 'You are not assigned to this unit for the specified class.');
        }

        // Get unit configuration
        $assessmentStructure = $unit->getAssessmentStructure();
        
        // Map assignment "type" values to assessment categories used in configuration/printing
        // - "assignment" + "exam"  => "theory"
        // - "practical"            => "practical"
        // - "project"              => "project"
        // There is no separate "oral" assignment type; oral is handled as part of practical via assessor tools.
        $typeToCategoryMap = [
            'assignment' => 'theory',
            'exam'       => 'theory',
            'practical'  => 'practical',
            'project'    => 'project',
        ];

        // Get all assignments for this unit, for this trainer and selected classes
        $assignmentsQuery = Assignment::where('unit_id', $unitId)
            ->where('trainer_id', $trainer->id)
            ->where('is_published', true)
            ->whereIn('class_id', $classIds);
        
        // Filter by category if provided
        if ($request->has('category') && $request->category) {
            $assignmentsQuery->where('type', $request->category);
        }
        
        $allAssignments = $assignmentsQuery->orderBy('created_at', 'asc')->get();
        
        // Group assignments by assessment category (theory/practical/project)
        $groupedAssignments = [
            'theory' => [],
            'practical' => [],
            'oral' => [],
            'project' => [],
        ];
        // Track raw type presence counts for quick checks in the view
        $typeCounts = [
            'assignment' => 0,
            'exam' => 0,
            'practical' => 0,
            'project' => 0,
        ];
        
        foreach ($allAssignments as $assignment) {
            if (isset($typeCounts[$assignment->type])) {
                $typeCounts[$assignment->type]++;
            }

            $category = $typeToCategoryMap[$assignment->type] ?? null;
            if ($category && isset($groupedAssignments[$category])) {
                $groupedAssignments[$category][] = $assignment;
            }
        }
        
        // Create a flat list with numbered assignments for display
        $assignments = [];
        $assignmentMap = []; // Map assignment ID to its display info
        
        foreach (['theory', 'practical', 'oral', 'project'] as $type) {
            $count = count($groupedAssignments[$type]);
            $requiredCount = $assessmentStructure[$type] ?? 0;
            
            // Only show if there are assignments or if required by config
            if ($count > 0 || $requiredCount > 0) {
                foreach ($groupedAssignments[$type] as $index => $assignment) {
                    $number = $index + 1;
                    $prefix = $type === 'theory' ? 'EXAM' : strtoupper($type);
                    $assignment->display_name = $prefix . $number;
                    $assignment->display_type = $type;
                    $assignments[] = $assignment;
                    $assignmentMap[$assignment->id] = [
                        'display_name' => $assignment->display_name,
                        'type' => $type,
                        'number' => $number,
                    ];
                }
            }
        }

        // Get all students in the selected classes for this unit
        $students = \App\Models\User::where('role', 'student')
            ->whereHas('enrollments', function($q) use ($classIds, $activeTerm) {
                $q->whereIn('class_id', $classIds)
                  ->where('status', 'active');
                if ($activeTerm) {
                    $q->whereHas('schoolClass', function($qc) use ($activeTerm) {
                        $qc->where('term_id', $activeTerm->id);
                    });
                }
            })
            ->select('id', 'name', 'email', 'admission_number')
            ->orderBy('name')
            ->get();

        // Get marks for each student and assignment, and calculate category averages
        $marksData = [];
        $categoryAverages = []; // Store category averages per student
        
        foreach ($students as $student) {
            $studentMarks = [];
            $categoryTotals = [
                'theory' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'practical' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'oral' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'project' => ['total' => 0, 'count' => 0, 'max_total' => 0],
            ];
            
            foreach ($allAssignments as $assignment) {
                $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->where('student_id', $student->id)
                    ->first();
                
                $marks = $submission ? $submission->marks : null;
                $markingStatus = $submission ? $submission->marking_status : 'not_marked';
                $percentage = null;
                
                // Handle absent students - they are assessed but with 0 marks
                if ($markingStatus === 'absent') {
                    $marks = 0; // Absent students get 0 marks but are still assessed
                    $percentage = 0;
                    
                    // Add to category totals (count as assessed with 0 marks)
                    $categoryKey = $typeToCategoryMap[$assignment->type] ?? null;
                    if ($categoryKey && isset($categoryTotals[$categoryKey])) {
                        $categoryTotals[$categoryKey]['total'] += 0; // 0 marks for absent
                        $categoryTotals[$categoryKey]['max_total'] += $assignment->total_marks;
                        $categoryTotals[$categoryKey]['count']++; // Count as assessed
                    }
                } elseif ($marks !== null && $assignment->total_marks > 0) {
                    $percentage = ($marks / $assignment->total_marks) * 100;
                    
                    // Add to category totals
                    $categoryKey = $typeToCategoryMap[$assignment->type] ?? null;
                    if ($categoryKey && isset($categoryTotals[$categoryKey])) {
                        $categoryTotals[$categoryKey]['total'] += $marks;
                        $categoryTotals[$categoryKey]['max_total'] += $assignment->total_marks;
                        $categoryTotals[$categoryKey]['count']++;
                    }
                }
                
                $studentMarks[$assignment->id] = [
                    'marks' => $marks,
                    'percentage' => $percentage,
                    // treat non-practical graded submissions as "marked" for readiness/printing purposes
                    'status' => $submission 
                        ? ($submission->marking_status && $submission->marking_status !== 'not_marked'
                            ? $submission->marking_status
                            : ($submission->status === 'graded' ? 'marked' : 'not_marked'))
                        : 'not_marked',
                    'submission' => $submission,
                    'total_marks' => $assignment->total_marks,
                ];
            }
            
            // Calculate category averages in percentage
            $categoryAverages[$student->id] = [];
            foreach (['theory', 'practical', 'oral', 'project'] as $type) {
                $cat = $categoryTotals[$type];
                if ($cat['count'] > 0 && $cat['max_total'] > 0) {
                    $categoryAverages[$student->id][$type] = [
                        'average_percentage' => ($cat['total'] / $cat['max_total']) * 100,
                        'count' => $cat['count'],
                        'total_marks' => $cat['total'],
                        'max_total' => $cat['max_total'],
                    ];
                } else {
                    $categoryAverages[$student->id][$type] = [
                        'average_percentage' => null,
                        'count' => 0,
                        'total_marks' => 0,
                        'max_total' => 0,
                    ];
                }
            }
            
            $marksData[$student->id] = $studentMarks;
        }

        /**
         * Practical / Oral breakdown per practical assessment (unit-level)
         *
         * For each practical assignment in this unit, we derive two percentages per student:
         * - Practical percentage (P1, P2, ...)
         * - Oral percentage (O1, O2, ...)
         *
         * These are based on assessor tool rows where possible. If no assessor tool exists,
         * we fall back to the overall submission marks as a single practical percentage.
         */
        $practicalAssignments = $allAssignments->where('type', 'practical')->values();
        $practicalTestCount = $practicalAssignments->count();

        $practicalUnitPercents = [];   // [studentId][testIndex]['P'|'O'] = percent
        $practicalUnitAverages = [];   // [studentId]['Pav'|'Oav'] = percent

        if ($practicalTestCount > 0 && $students->count() > 0) {
            $studentIds = $students->pluck('id');

            // Initialise container
            foreach ($students as $student) {
                $practicalUnitPercents[$student->id] = [];
            }

            foreach ($practicalAssignments as $testIndex => $assignment) {
                // Load assessor tool structure if available
                $structure = \App\Models\AssessorToolStructure::where('assignment_id', $assignment->id)->first();

                $pMax = 0.0;
                $oMax = 0.0;

                if ($structure && is_array($structure->structure ?? null)) {
                    $sections = $structure->structure['sections'] ?? [];

                    foreach ($sections as $row) {
                        $rowId = $row['row_identifier'] ?? '';
                        $marksAvailable = (float) ($row['marks_available'] ?? 0);
                        if (!$rowId || $marksAvailable <= 0) {
                            continue;
                        }

                        $isPractical = preg_match('/practical/i', $rowId);
                        $isOral = preg_match('/oral/i', $rowId);

                        if ($isPractical) {
                            $pMax += $marksAvailable;
                        } elseif ($isOral) {
                            $oMax += $marksAvailable;
                        }
                    }
                }

                // Fallback if no detailed structure: treat whole assignment as practical only
                if ($pMax <= 0 && $oMax <= 0) {
                    $pMax = (float) ($assignment->total_marks ?: 100);
                }

                // Get submissions for this assignment for the unit's students
                $submissions = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->whereIn('student_id', $studentIds)
                    ->get()
                    ->keyBy('student_id');

                // Get detailed marks if we have structure
                $marksBySubmission = collect();
                if ($structure && ($pMax > 0 || $oMax > 0)) {
                    $submissionIds = $submissions->pluck('id')->filter()->values();
                    if ($submissionIds->isNotEmpty()) {
                        $marksBySubmission = \App\Models\AssessorToolMark::where('assignment_id', $assignment->id)
                            ->whereIn('assignment_submission_id', $submissionIds)
                            ->get()
                            ->groupBy('assignment_submission_id');
                    }
                }

                foreach ($students as $student) {
                    $pTotal = 0.0;
                    $oTotal = 0.0;
                    $pPercent = null;
                    $oPercent = null;

                    $submission = $submissions->get($student->id);

                    if ($structure && ($pMax > 0 || $oMax > 0) && $submission && $marksBySubmission->has($submission->id)) {
                        // Aggregate P and O from assessor tool marks
                        foreach ($marksBySubmission->get($submission->id) as $mark) {
                            $rowId = $mark->row_identifier ?? '';
                            if (preg_match('/practical/i', $rowId)) {
                                $pTotal += (float) $mark->marks_obtained;
                            } elseif (preg_match('/oral/i', $rowId)) {
                                $oTotal += (float) $mark->marks_obtained;
                            }
                        }

                        // If we failed to classify P/O from the structure (e.g. legacy
                        // row identifiers without PRACTICAL / ORAL), fall back to the
                        // overall submission marks so the student still gets a percentage.
                        if ($pTotal == 0.0 && $oTotal == 0.0 && $submission->marks !== null && (float) $submission->marks > 0) {
                            $pTotal = (float) $submission->marks;
                            $oTotal = 0.0;
                        }

                        if ($pMax > 0) {
                            $pPercent = ($pTotal / $pMax) * 100;
                        }
                        if ($oMax > 0) {
                            $oPercent = ($oTotal / $oMax) * 100;
                        }
                    } elseif ($submission && $submission->marks !== null && $pMax > 0) {
                        // Fallback: use overall submission marks as a single practical percentage
                        $pTotal = (float) $submission->marks;
                        $pPercent = ($pTotal / $pMax) * 100;
                    }

                    $practicalUnitPercents[$student->id][$testIndex] = [
                        'P' => $pPercent,
                        'O' => $oPercent,
                    ];
                }
            }

            // Compute per-student averages across all practical tests
            foreach ($students as $student) {
                $entries = $practicalUnitPercents[$student->id] ?? [];
                $pValues = [];
                $oValues = [];
                foreach ($entries as $entry) {
                    if (isset($entry['P']) && $entry['P'] !== null) {
                        $pValues[] = $entry['P'];
                    }
                    if (isset($entry['O']) && $entry['O'] !== null) {
                        $oValues[] = $entry['O'];
                    }
                }

                $pAvg = count($pValues) > 0 ? array_sum($pValues) / count($pValues) : null;
                $oAvg = count($oValues) > 0 ? array_sum($oValues) / count($oValues) : null;

                $practicalUnitAverages[$student->id] = [
                    'Pav' => $pAvg,
                    'Oav' => $oAvg,
                ];
            }
        }

        // Get institution settings for letterhead
        $institutionSettings = \App\Models\Setting::where('key', 'like', 'institution.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        // Simple presence flags so the Blade view can decide what to display
        $assessmentPresence = [
            'has_assignment' => $typeCounts['assignment'] > 0,
            'has_exam' => $typeCounts['exam'] > 0,
            'has_practical' => $typeCounts['practical'] > 0,
            'has_project' => $typeCounts['project'] > 0,
        ];

        return view('trainer.marks.print-unit', compact(
            'unit', 
            'assignments', 
            'students', 
            'marksData', 
            'activeTerm',
            'groupedAssignments',
            'categoryAverages',
            'assessmentStructure',
            'institutionSettings',
            'trainer',
            'assessmentPresence',
            'practicalUnitPercents',
            'practicalUnitAverages',
            'selectedClass'
        ));
    }

    /**
     * Legacy PoE marks update endpoint (no longer used).
     * Kept only to avoid route errors; redirects with message.
     */
    public function update(Request $request, PoeSubmission $submission)
    {
        return redirect()
            ->route('trainer.marks.index')
            ->with('error', 'Direct PoE submission marking has been disabled. Please use assignment marks instead.');
    }
}
