<?php

namespace App\Http\Middleware;

use App\Models\AuditLog;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class AuditLogMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        /** @var \Symfony\Component\HttpFoundation\Response $response */
        $response = $next($request);

        // Only log authenticated user actions
        if (!auth()->check()) {
            return $response;
        }

        // Only log state-changing requests
        if (!in_array(strtoupper($request->method()), ['POST', 'PUT', 'PATCH', 'DELETE'])) {
            return $response;
        }

        // Avoid logging audit log page interactions themselves to reduce noise
        $routeName = optional($request->route())->getName();
        if ($routeName && str_starts_with($routeName, 'admin.audit-logs.')) {
            return $response;
        }

        try {
            // Build action label
            $action = $routeName ?: ($request->method() . ' ' . $request->path());

            // Try to infer a primary model from route parameters
            $modelType = null;
            $modelId = null;
            $routeParams = optional($request->route())->parameters() ?? [];

            foreach ($routeParams as $param) {
                if (is_object($param) && method_exists($param, 'getKey')) {
                    $modelType = get_class($param);
                    $modelId = $param->getKey();
                    break;
                }
            }

            // Capture request data, excluding sensitive fields and large payloads
            $input = $request->except([
                'password',
                'password_confirmation',
                '_token',
                '_method',
                'current_password',
            ]);

            // Truncate very long string values
            array_walk_recursive($input, function (&$value) {
                if (is_string($value) && strlen($value) > 500) {
                    $value = substr($value, 0, 500) . '...';
                }
            });

            AuditLog::create([
                'user_id' => auth()->id(),
                'action' => $action,
                'model_type' => $modelType,
                'model_id' => $modelId,
                'changes' => $input ?: null,
                'ip_address' => $request->ip(),
                'description' => 'HTTP ' . $request->method() . ' ' . $request->fullUrl() . ' (status ' . $response->getStatusCode() . ')',
                'created_at' => now(),
            ]);
        } catch (\Throwable $e) {
            // Never break the app because of audit logging
        }

        return $response;
    }
}

