<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Services\ActiveTermService;

class PoeSubmission extends Model
{
    use HasFactory;

    protected $fillable = [
        'student_id',
        'unit_id',
        'class_id',
        'status',
        'version',
        'submitted_at',
        'notes',
    ];

    protected $casts = [
        'submitted_at' => 'datetime',
        'version' => 'integer',
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        // Add global scope for active term (can be bypassed with withoutGlobalScope)
        static::addGlobalScope('activeTerm', function ($query) {
            $activeTermId = ActiveTermService::getActiveTermId();
            if ($activeTermId) {
                $query->join('classes', 'poe_submissions.class_id', '=', 'classes.id')
                      ->where('classes.term_id', $activeTermId)
                      ->select('poe_submissions.*');
            }
        });
    }

    // Relationships
    public function student()
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    public function unit()
    {
        return $this->belongsTo(Unit::class);
    }

    public function schoolClass()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    public function evidence()
    {
        return $this->hasMany(PoeEvidence::class);
    }

    public function validation()
    {
        return $this->hasOne(Validation::class);
    }

    // Helper methods
    public function isDraft()
    {
        return $this->status === 'draft';
    }

    public function isSubmitted()
    {
        return in_array($this->status, ['submitted', 'under_review', 'trainer_reviewed', 'hod_approved', 'exams_verified', 'validator_approved']);
    }

    public function isApproved()
    {
        return $this->status === 'validator_approved';
    }

    public function isRejected()
    {
        return $this->status === 'rejected';
    }

    /**
     * Get status badge color classes
     */
    public function getStatusBadgeClasses()
    {
        return match($this->status) {
            'draft' => 'bg-gray-100 text-gray-800',
            'submitted' => 'bg-blue-100 text-blue-800',
            'under_review' => 'bg-yellow-100 text-yellow-800',
            'trainer_reviewed' => 'bg-purple-100 text-purple-800',
            'hod_approved' => 'bg-indigo-100 text-indigo-800',
            'exams_verified' => 'bg-cyan-100 text-cyan-800',
            'validator_approved' => 'bg-green-100 text-green-800',
            'rejected' => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    /**
     * Get human-readable status
     */
    public function getStatusLabel()
    {
        return str_replace('_', ' ', ucfirst($this->status));
    }

    /**
     * Scope to bypass active term filter (for admin/historical views)
     */
    public function scopeAllTerms($query)
    {
        return $query->withoutGlobalScope('activeTerm');
    }
}
