<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Unit extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'code',
        'description',
        'department_id',
        'classification',
        'assessment_structure',
        'configured_by',
        'configured_at',
    ];

    protected $casts = [
        'assessment_structure' => 'array',
        'configured_at' => 'datetime',
    ];

    // Relationships
    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    public function classes()
    {
        return $this->belongsToMany(SchoolClass::class, 'class_unit', 'unit_id', 'class_id')
                    ->withTimestamps();
    }

    public function poeSubmissions()
    {
        return $this->hasMany(PoeSubmission::class);
    }

    public function trainers()
    {
        return $this->belongsToMany(User::class, 'trainer_unit', 'unit_id', 'trainer_id')
                    ->where('role', 'trainer')
                    ->withTimestamps();
    }

    public function assignments()
    {
        return $this->hasMany(Assignment::class);
    }

    public function practicalEvidences()
    {
        return $this->hasMany(PracticalEvidence::class);
    }

    public function configuredBy()
    {
        return $this->belongsTo(User::class, 'configured_by');
    }

    /**
     * Get the required count for a specific assessment type
     */
    public function getRequiredAssessmentCount(string $type): int
    {
        $structure = $this->assessment_structure ?? [];
        return $structure[$type] ?? 0;
    }

    /**
     * Check if unit is configured
     */
    public function isConfigured(): bool
    {
        return !is_null($this->classification) && !is_null($this->assessment_structure);
    }

    /**
     * Get assessment structure with defaults
     */
    public function getAssessmentStructure(): array
    {
        return $this->assessment_structure ?? [
            'theory' => 0,
            'practical' => 0,
            'oral' => 0,
            'project' => 0,
        ];
    }
}
