@extends('layouts.admin')

@section('title', 'Term Management')
@section('page-title', 'Term Management')

@section('content')
    <div class="mb-6 flex flex-col sm:flex-row sm:items-center sm:justify-between">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Academic Terms</h1>
            <p class="mt-1 text-sm text-gray-600">Manage academic terms and set the active term</p>
        </div>
        <a href="{{ route('admin.terms.create') }}" class="mt-4 sm:mt-0 inline-flex items-center px-4 py-2 bg-indigo-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-indigo-700 focus:bg-indigo-700 active:bg-indigo-900 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 transition ease-in-out duration-150">
            <svg class="mr-2 h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
            </svg>
            Create New Term
        </a>
    </div>

    <!-- Active Term Banner -->
    @if($activeTerm)
        <div class="bg-indigo-50 border-l-4 border-indigo-500 p-4 mb-6 rounded">
            <div class="flex items-center justify-between">
                <div class="flex items-center">
                    <svg class="h-5 w-5 text-indigo-500 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    <div>
                        <p class="text-sm font-semibold text-indigo-800">Active Term: {{ $activeTerm->name }}</p>
                        <p class="text-xs text-indigo-600">
                            {{ $activeTerm->start_date->format('M d, Y') }} - {{ $activeTerm->end_date->format('M d, Y') }}
                            ({{ $activeTerm->getDurationInDays() }} days)
                        </p>
                    </div>
                </div>
                <form action="{{ route('admin.terms.close', $activeTerm) }}" method="POST" class="inline">
                    @csrf
                    <button type="submit" onclick="return confirm('Are you sure you want to close this term? This will make it read-only.')" 
                            class="text-xs px-3 py-1 bg-indigo-600 text-white rounded hover:bg-indigo-700">
                        Close Term
                    </button>
                </form>
            </div>
        </div>
    @else
        <div class="bg-yellow-50 border-l-4 border-yellow-500 p-4 mb-6 rounded">
            <div class="flex items-center">
                <svg class="h-5 w-5 text-yellow-500 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                </svg>
                <p class="text-sm font-semibold text-yellow-800">No active term. Please activate a term to enable system operations.</p>
            </div>
        </div>
    @endif

    <!-- Terms Table -->
    <div class="bg-white shadow rounded-lg overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Term Name</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date Range</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Duration</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Classes</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($terms as $term)
                        <tr class="hover:bg-gray-50 {{ $term->isActive() ? 'bg-indigo-50' : '' }}">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm font-medium text-gray-900">{{ $term->name }}</div>
                                @if($term->isActive())
                                    <div class="text-xs text-indigo-600 font-semibold">Currently Active</div>
                                @endif
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm text-gray-900">{{ $term->start_date->format('M d, Y') }}</div>
                                <div class="text-sm text-gray-500">to {{ $term->end_date->format('M d, Y') }}</div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {{ $term->getDurationInDays() }} days
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full 
                                    @if($term->status == 'active') bg-green-100 text-green-800
                                    @elseif($term->status == 'draft') bg-gray-100 text-gray-800
                                    @elseif($term->status == 'completed') bg-blue-100 text-blue-800
                                    @else bg-gray-100 text-gray-800
                                    @endif">
                                    {{ ucfirst($term->status) }}
                                </span>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {{ $term->classes()->count() }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                <div class="flex items-center justify-end space-x-2">
                                    <a href="{{ route('admin.terms.show', $term) }}" class="text-indigo-600 hover:text-indigo-900">View</a>
                                    
                                    @if(!$term->isClosed())
                                        <a href="{{ route('admin.terms.edit', $term) }}" class="text-indigo-600 hover:text-indigo-900">Edit</a>
                                    @endif

                                    @if(!$term->isActive() && !$term->isClosed())
                                        <form action="{{ route('admin.terms.activate', $term) }}" method="POST" class="inline">
                                            @csrf
                                            <button type="submit" class="text-green-600 hover:text-green-900" 
                                                    onclick="return confirm('Activate this term? The current active term will be closed.')">
                                                Activate
                                            </button>
                                        </form>
                                    @endif

                                    @if($term->isDraft())
                                        <form action="{{ route('admin.terms.destroy', $term) }}" method="POST" class="inline" 
                                              onsubmit="return confirm('Are you sure you want to delete this term?');">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="text-red-600 hover:text-red-900">Delete</button>
                                        </form>
                                    @endif
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="px-6 py-4 text-center text-sm text-gray-500">
                                No terms found. <a href="{{ route('admin.terms.create') }}" class="text-indigo-600 hover:text-indigo-900">Create your first term</a>
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>

    <!-- Information Box -->
    <div class="mt-6 bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div class="flex">
            <div class="flex-shrink-0">
                <svg class="h-5 w-5 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                </svg>
            </div>
            <div class="ml-3">
                <h3 class="text-sm font-medium text-blue-800">About Term Management</h3>
                <div class="mt-2 text-sm text-blue-700">
                    <ul class="list-disc list-inside space-y-1">
                        <li>Only one term can be active at a time</li>
                        <li>Activating a new term automatically closes the previous active term</li>
                        <li>Closed terms (completed/archived) are read-only for audit purposes</li>
                        <li>All academic data is scoped to the active term</li>
                        <li>Historical data remains accessible for reporting</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
@endsection

